// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit tTransfer; // Threads for news and mail transfers.

// ----------------------------------------------------------------------------
// Contains all threads, which handle the transfer of news and mails from/to
// remote servers.
// ----------------------------------------------------------------------------

interface

uses Windows, SysUtils, Classes, tBase, cClientNNTP;

type
  TThreadPop3Fetch = class( TTaskThread )
    private
      FServer, FPort, FUser, FPass,
      FDestUsername, FFilterSection, FLeaveServer: String;
      FSSLMode, FSSLVerifyLevel: Integer;
      FSSLCaFile: String; {MG}{SSL}
      procedure FetchMailFromServer;
    protected
      procedure Execute; override;
    public
{MG}{SSL}
     constructor Create( Const AServer, APort, AUser, APass, ADestUsername, AFilterSection,
        ALeaveServer: String; Const ASSLMode, ASSLVerifyLevel: Integer; Const ASSLCaFile: String );
{/SSL}
     destructor destroy; override;
  end;

 TThreadSmtpSend = class( TTaskThread )
    private
      FServer, FPort, FUser, FPass: String;
      FSSLMode, FSSLVerifyLevel: Integer; FSSLCaFile: String; {MG}{SSL}
      FScriptAuth: Boolean; {JW} {SMTP Auth 08.12.00}
      FFromSelect, FToSelect: String;
      ToSendSnapshot: TStringList;
    protected
      procedure SendMail;
      procedure Execute; override;
    public
      constructor Create( Const AServer, APort, AUser, APass: String;
                          Const AFromSelect, AToSelect: String;
                          Const AScriptAuth: Boolean;
                          Const ASSLMode, ASSLVerifyLevel: Integer;
                          Const ASSLCaFile: String );
      destructor destroy; override;
  end;

  TThreadPop3AndSmtp = class( TTaskThread )
    private
      Killme: Boolean;
      ThreadBez, ServerList: String;
      Mode: Integer; // 0=POP3+SMTP, 1=POP3, 2=SMTP
    protected
      procedure MailFromServer( LfdServer: Integer );
      procedure SendMail;
      procedure Execute; override;
    public
      constructor Create( Const AServerList: String; Const AMode: Integer );
  end;

  TThreadNewsJobs = class( TTaskThread )
    private
      LfdServer : Integer;
      NewsClient: TClientNNTP;
      procedure NukePostedArticle( Const PostFile, ArtText, DestGroup, Reason,
         DestSrv: String; Const PostedOK: Boolean );
      procedure PostArticle( PostFile: String );
    protected
      procedure Execute; override;
    public
      constructor Create( Const ALfdServer: Integer );
      destructor destroy; override;
  end;

  TThreadNewsPostSimulate = class( TTaskThread )
    private
      ServerList: String;
    protected
      procedure Execute; override;
    public
      constructor Create( Const AServerList: String );
    destructor destroy; override;
  end;

implementation

uses Global, Config, uTools, uCRC32, cPCRE, cPasswordFile, cAccount, cArticle,
     cArtFiles, cNewsJobs, cClientBase, cClientPOP3, cClientSMTP, uDateTime,
     IniFiles, cStdForm, cLogFile,
     cMailRouter; //JW //NNTPINFOMAIL

// ----------------------------------------------------- TThreadPop3Fetch -----

procedure TThreadPop3Fetch.FetchMailFromServer;
var  LeaveOnServer, bFirst, bStop: Boolean;
     LfdServer, MaxMails: Integer;
     s: String;
begin
   // Settings
   LfdServer := CfgHamster.Pop3ServerIndexOf[ FServer ];
   LeaveOnServer := Def_LeaveMailsOnServer;
   MaxMails := 0;
   If LfdServer >= 0 then begin
      With TIniFile.Create( CfgHamster.Pop3ServerPath[LfdServer] + SRVFILE_INI ) do try
         s := ReadString( 'POP3', 'LeaveOnServer', '?' );
         If (s ='1') or (s='0') then LeaveOnServer := (s = '1');
         MaxMails := ReadInteger('POP3', 'GetMaxMailsPerSession', MaxMails)
      finally
         Free
      end
   end;
   If FLeaveServer = '1' then LeaveOnServer := true
   else If FLeaveServer = '0' then LeaveOnServer := false;

   // Let's do it...
   bFirst := true;
   With TClientPOP3.Create( FServer, FPort, FUser, FPass,
                            FDestUserName, FFilterSection,
                            FSSLMode, FSSLVerifyLevel, FSSLCaFile )
   do try
      Repeat
         TLog( LOGID_INFO, TrGl(kLog, 'Info.Connecting', 'Connecting ...' ) );
         Connect;
         if ShutdownReq or Terminated then Exit;
         If State in [csERROR, csDISCONNECTED] then exit;
         If bFirst then begin
            GetServerInfos;
            bFirst := false
         end;
         if Terminated or ShutdownReq then Exit;
         If State in [csERROR, csDISCONNECTED] then Exit;
         bStop := Not GetNewMails( LeaveOnServer, FDestUsername, FFilterSection, MaxMails);
         TLog( LOGID_INFO, TrGl(kLog, 'Info.DisConnecting', 'Disconnecting ...' ) );
         Disconnect
      until bStop or (MaxMails=0)
   finally
      Free
   end
   
end;

procedure TThreadPop3Fetch.Execute;
var  OK: Boolean;
     s : String;
     UID: Integer;
begin
{JW} {Tasks}
     wait;
{JW}
     TLog( LOGID_SYSTEM, TrGl(klog, 'System.Start', 'Start' ));
     OK := True;

     // check given servername
     if FServer='' then begin
        TLog( LOGID_ERROR, TrGl(kLog, 'Error.MissingServername', 'Missing servername!'));
        OK := False;
     end;

     // use default-port if not given
     if FPort='' then FPort := 'pop3';

     // check given user
     CfgAccounts.Lock;
     try
        if FDestUsername='' then UID:=ACTID_ADMIN
                            else UID:=CfgAccounts.Users.IDOf(FDestUsername);
        if UID=ACTID_INVALID then begin
           TLog( LOGID_ERROR, TrGlF(kLog, 'Error.POP3.UnknownUsername', 'Unknown username "%s"!', FDestUsername) );
           OK := False;
        end else With CfgAccounts.Users.Find(UID) do begin
           if not ( HasMailbox or HasIMAPbox ) then begin {IMAP}
              TLog( LOGID_ERROR, TrGlF(kLog, 'Error.POP3.UserWithoutMailbox', 'User "%s" does not have a mailbox!', FDestUsername ) );
              OK := False;
           end;
        end
     finally
        CfgAccounts.UnLock
     end;

     // resolve username and password
     if OK and ( (FUser='') or (copy(FUser,1,1)='$') ) and (FPass='') then begin
        if FUser='' then s := FServer // server-password
                    else s := FUser;  // script-password ($0..$99)
        if not PasswordFile.UsePassword( s, FUser, FPass ) then begin
           TLog( LOGID_ERROR, TrGlF(kLog, 'Error.POP3.MissingUsernamePW',
              'Missing username/password for "%s"!', s) );
           OK := False;
        end;
     end;

     // get mails from selected server
     if OK then FetchMailFromServer;

     TLog( LOGID_SYSTEM, TrGl(kLog, 'System.end', 'End'));
end;

{JW} {SSL}
constructor TThreadPop3Fetch.Create( Const AServer, APort, AUser, APass,
                                     ADestUsername, AFilterSection,
                                     ALeaveServer: String;
                                     Const ASSLMode, ASSLVerifyLevel: Integer;
                                     Const ASSLCaFile: String );
{JW}
var  s: String;
     LfdServer: Integer;
begin
   s := AServer;
   if (copy(AUser,1,1)='$') and (APass='') and (length(AUser)<=3) then s:=s+','+AUser;
   inherited Create( '{fetchmail ' + s + '}' );

   FServer       := AServer;
   FPort         := APort;
   FUser         := AUser;
   FPass         := APass;

   {MG}{SSL}
   FSSLMode        := ASSLMode;
   FSSLVerifyLevel := ASSLVerifyLevel;
   FSSLCaFile      := ASSLCaFile;
   {/SSL}

   FDestUsername  := ADestUsername;
   FFilterSection := AFilterSection;
   FLeaveServer   := ALeaveServer;

   LfdServer := CfgHamster.Pop3ServerIndexOf [FServer+','+FPort];
   If LfdServer >= 0 then begin
      If FDestusername = '' then FDestUsername := CfgHamster.Pop3ServerLocalUser[LfdServer];
      If FFilterSection = '' then FFilterSection := CfgHamster.Pop3ServerFiltersection[LfdServer]
   end;
end;

destructor TThreadPop3Fetch.destroy;
begin
   Mainlog(LOGID_SYSTEM, TrGlF(kLog, 'Fetchmail.end', 'fetchmail %s ready', FServer) );
   inherited;
end;

// ------------------------------------------------------ TThreadSmtpSend -----

procedure TThreadSmtpSend.SendMail;
var  SendMailClient: TClientSMTP;
     MailFile      : String;
begin
     // connect to server
     TLog( LOGID_INFO, TrGl(kLog, 'Info.Connecting', 'Connecting ...' ) );
     {MG}{SSL}
     SendMailClient := TClientSMTP.Create( FServer, FPort, FUser, FPass,
                                           FSSLMode, FSSLVerifyLevel, FSSLCaFile );
     {/SSL}
     SendmailClient.ScriptAuth:=FScriptAuth; {JW} {SMTP Auth 08.12.00}
     SendMailClient.Connect;

     // send mails
     while ToSendSnapshot.Count>0 do begin
        if Terminated then break;
        if ShutdownReq then break;
        if SendMailClient.State in [csERROR, csDISCONNECTED] then break;

        MailFile := ToSendSnapshot[0];
        ToSendSnapshot.Delete( 0 );

        TLog( LOGID_INFO, TrGlF(kLog, 'Info.SMTP.MailSending','Send mail "%s"',
             ExtractFilename(MailFile)));
        if not SendMailClient.SendMailfile( MailFile ) then break;
        IncCounter( CntOutboxChk, 1 );
     end;

     // disconnect from server
     TLog( LOGID_INFO, TrGl(kLog, 'Info.Disconnecting', 'Disconnecting ...' ) );
     try SendMailClient.Disconnect; except end;
     try SendMailClient.Free;       except end;
     IncCounter( CntOutboxChk, 1 );
end;

procedure TThreadSmtpSend.Execute;
var  SR    : TSearchRec;
     res, LfdMail: Integer;
     rexFrom, rexTo: TPCRE;
     s: String;
     bFromSel, bToSel, FromIsOK, ToIsOK, OK: Boolean;
begin
   wait;
   TLog( LOGID_SYSTEM, TrGl(kLog, 'System.Start', 'Start' ));

   // check given servername
   if FServer='' then begin
      TLog( LOGID_ERROR, TrGl(kLog, 'Error.MissingServername', 'Missing servername!' ) );
      exit;
   end;

   // Check given Selection { NEU! }
   With TPCRE.Create( True, PCRE_CASELESS ) do try
      If FFromSelect > '' then try
         Compile( PChar(FFromSelect) )
      except
         TLog( LOGID_ERROR, TrGlF(kLog, 'Error.ServerSelection.RegExpError',
            'Error in From-Selection-Regexp "%s"!', FFromSelect) );
         exit;
      end;
      If FToSelect > '' then try
         Compile( PChar(FToSelect) )
      except
         TLog( LOGID_ERROR, TrGlF(kLog, 'Error.ServerSelectionTo.RegExpError',
            'Error in To-Selection-Regexp "%s"!', FToSelect) );
         exit;
      end
   finally
      free
   end;

   // use default-port if not given
   if FPort='' then FPort := 'smtp';

   // block other smtpsend-threads
   EnterCriticalSection( CS_LOCK_MAILOUT_USE );

   // create a snapshot of all mails to send
   try
      // prevent local smtp-server from adding mails while building snapshot
      EnterCriticalSection( CS_LOCK_MAILOUT_ADD );

      // build a snapshot of all mails to send
      ToSendSnapshot := TStringList.Create;
      res := FindFirst( PATH_MAIL_OUT + '*.'
         + CfgIni.ReadString( 'Setup', 'mail.ext.out', 'msg' ), faAnyFile-faDirectory, SR );
      while res=0 do begin
         ToSendSnapshot.Add( PATH_MAIL_OUT + SR.Name );
         res := FindNext( SR );
      end;
      FindClose( SR );
   finally
      // now we have a reliable list of mailfiles to send, so local
      // smtp-server may continue to add new mails in mail.out
      LeaveCriticalSection( CS_LOCK_MAILOUT_ADD );
   end;

   // filter out unselected mails
   bFromSel := (FFromSelect<>'') and (FFromSelect<>'.*');
   bToSel   := (FToSelect<>'') and (FToSelect<>'.*');
   if (ToSendSnapshot.Count > 0) and (bFromSel or bToSel) then begin
      If bFromSel then rexFrom := TPCRE.Create( True, PCRE_CASELESS ) else rexFrom := NIL;
      If bToSel   then rexTo   := TPCRE.Create( True, PCRE_CASELESS ) else rexTo := NIL;
      try
         If bFromSel then rexFrom.Compile( PChar(FFromSelect) );
         If bToSel then rexTo.Compile( PChar(FToSelect) );
         LfdMail := 0;
         while LfdMail < ToSendSnapShot.Count do begin
            ok := false;
            FromIsOK := Not bFromSel;
            ToIsOK := Not bToSel;
            try
               With TTextReader.Create( ToSendSnapShot[LfdMail], 2048 ) do try
                  While Not (eof or (FromIsOK and ToIsOK)) do begin
                     s := ReadLine;
                     If lowercase(copy(s,1,11))='!mail from:' then begin
                        If Not FromIsOK then begin
                           System.Delete( s, 1, 11 );
                           s := TrimWhSpace( s );
                           if rexFrom.Exec( PChar(s), 0 ) then begin
                              FromIsOK := True;
                              TLog( LOGID_DEBUG, TrGlF(kLog, 'Debug.SMTP.FromMatch.OK',
                                 'From-Match(%s,%s) OK', [FFromSelect, s] ));
                           end else begin
                              TLog( LOGID_DEBUG, TrGlF(kLog, 'Debug.SMTP.FromMatch.failed',
                                 'From-Match(%s,%s) failed', [FFromSelect, s] ));
                              Break
                           end
                        end
                     end else
                     If lowercase(copy(s,1,9))='!rcpt to:' then begin
                        If Not ToIsOK then begin
                           System.Delete( s, 1, 9 );
                           s := TrimWhSpace( s );
                           if rexTo.Exec( PChar(s), 0 ) then begin
                              ToIsOK := True;
                              TLog( LOGID_DEBUG, TrGlF(kLog, 'Debug.SMTP.ToMatch.OK',
                                 'To-Match(%s,%s) OK', [FFromSelect, s] ));
                           end
                        end
                     end else begin
                        Break
                     end
                  end; // While
                  If bToSel and (Not ToIsOK) then begin
                     TLog( LOGID_DEBUG, TrGlF(kLog, 'Debug.SMTP.ToMatch.failed2',
                        'All To-Match(%s) failed', [FToSelect, s] ));
                  end
               finally
                  free
               end;
               ok := FromIsOK and ToIsOK
            except
               on E:Exception do begin
                  TLog( LOGID_ERROR, TrGlF(kLog, 'Error.SMTP.Checking',
                     'Error checking "%s"!', ToSendSnapShot[LfdMail] ) );
                  TLog( LOGID_ERROR, TrGl(kLog, 'Error', 'Error') + '=' + E.Message );
               end
            end;
            if OK then inc( LfdMail )
                  else ToSendSnapShot.Delete( LfdMail );

         end;
      finally
         If bFromSel then rexFrom.Free;
         If bToSel then rexTo.free
      end
   end;

   // send mails to selected server
   if ToSendSnapshot.Count > 0 then begin

      // resolve username and password
      if ( (FUser='') or (copy(FUser,1,1)='$') ) and (FPass='') then begin
         if FUser='' then begin
            s := FServer; // server-password
            TLog( LOGID_DEBUG, 'Use passwort from gui '+FServer );
         end else begin
            s := FUser;  // script-password ($0..$99)
            TLog( LOGID_DEBUG, 'Use passwort from variable '+FUser );
         end;
         if not PasswordFile.UsePassword( s, FUser, FPass ) then begin
            TLog( LOGID_DEBUG, TrGlF(kLog, 'Debug.SMTP.MissingUsernamePW',
               'Missing username/password for "%s"!', s) );
         end;
      end;

      TLog( LOGID_DEBUG, 'Use username '+FUser );
      if FPass=''
         then TLog( LOGID_DEBUG, 'Password is empty' )
         else TLog( LOGID_DEBUG, 'Password is not empty' );

      // send mails
      SendMail;

   end else begin
      TLog( LOGID_INFO, TrGl(kLog, 'Info.NoMailsToSend', 'No mail to send.' ) );
   end;

   // release other smtpsend-threads
   LeaveCriticalSection( CS_LOCK_MAILOUT_USE );

   ToSendSnapshot.Free;
   TLog( LOGID_SYSTEM, TrGl(kLog, 'System.End', 'End' ));
end;

constructor TThreadSmtpSend.Create( Const AServer, APort, AUser, APass: String;
   Const AFromSelect, AToSelect: String;
   Const AScriptAuth:Boolean;
   Const ASSLMode, ASSLVerifyLevel: Integer;
   Const ASSLCaFile: String );
begin
   inherited Create( '{sendmail ' + AServer + '}' );
   FScriptAuth := AScriptAuth;
   FServer     := AServer;
   FPort       := APort;
   FUser       := AUser;
   FPass       := APass;
   FFromSelect := AFromSelect;
   FToSelect   := AToSelect;
   FSSLMode        := ASSLMode;
   FSSLVerifyLevel := ASSLVerifyLevel;
   FSSLCaFile      := ASSLCaFile;
end;

destructor TThreadSmtpSend.destroy;
begin
   Mainlog(LOGID_SYSTEM, TrGlF(kLog, 'SMTPSend.end', 'Send mail over %s (SMTP) ready', FServer ) );
   inherited;
end;

// --------------------------------------------------- TThreadPop3AndSmtp -----

procedure TThreadPop3AndSmtp.MailFromServer( LfdServer: Integer );
Var LeaveOnServer: Boolean;
    bFirst, bStop: Boolean;
    MaxMails     : Integer;
    s: String;
begin
   LeaveOnServer := Def_LeaveMailsOnServer;
   With TIniFile.Create( CfgHamster.Pop3ServerPath[LfdServer] + SRVFILE_INI ) do try
      s := ReadString( 'POP3', 'LeaveOnServer', '?' );
      If (s ='1') or (s='0') then LeaveOnServer := (s = '1');
      MaxMails := ReadInteger('POP3', 'GetMaxMailsPerSession', 0)
   finally
      Free
   end;
   bFirst := true;
   With CfgHamster do begin
     {MG}{SSL}
      With TClientPOP3.Create(
         Pop3ServerName[LfdServer], Pop3ServerPort[LfdServer],
         Pop3ServerUser[LfdServer], Pop3ServerPass[LfdServer],
         Pop3ServerLocalUser[LfdServer], Pop3ServerFilterSection[LfdServer],
         Pop3ServerMode[LfdServer],
         Pop3ServerVerify[LfdServer],
         Pop3ServerCaFile[LfdServer])
     {/SSL}
      do try
         Repeat
            TLog( LOGID_INFO, TrGl(kLog, 'Info.Connecting', 'Connecting ...' ) );
            Connect;
            If State in [csERROR, csDISCONNECTED] then exit;
            If bFirst then begin
               GetServerInfos;
               bFirst := false
            end;
            if Terminated or ShutdownReq then Exit;
            If State in [csERROR, csDISCONNECTED] then Exit;
            bStop := Not GetNewMails( LeaveOnServer, '', '', MaxMails);
            TLog( LOGID_INFO, TrGl(kLog, 'Info.DisConnecting', 'Disconnecting ...' ) );
            Disconnect
         until bStop or (MaxMails=0)
      finally
         Free
      end
   end
end;

procedure TThreadPop3AndSmtp.SendMail;

   Function SearchSMTPServer: Integer;
   Var s: String; i: Integer;
   begin
      Result := CfgHamster.SmtpServerIndexOf[Def_SmtpServer];
      if Result >= 0 then begin
         if ServerList>'' then begin
            s := CfgHamster.SmtpServerName[Result] + ',' + CfgHamster.SmtpServerPort[Result];
            if Pos( ';' + LowerCase(s) + ';', ';' + LowerCase(ServerList) + ';' ) = 0
               then Result:=-1
         end
      end;
      if Result < 0 then begin
         if ServerList='' then begin
            Result := 0
         end else begin
            for i:=0 to CfgHamster.SmtpServerCount-1 do If Result < 0 then begin
               s := CfgHamster.SmtpServerName[i] + ',' + CfgHamster.SmtpServerPort[i];
               if Pos( ';' + LowerCase(s) + ';', ';' + LowerCase(ServerList) + ';' ) > 0
                  then Result := i
            end
         end
      end;
   end;

var  SR : TSearchRec;
     SendMailClient: TClientSMTP;
     LfdServer: Integer;
begin
   If FindFirst( PATH_MAIL_OUT+'*.'+CfgIni.ReadString( 'Setup', 'mail.ext.out', 'msg' ),
      faAnyFile, SR ) = 0
   then begin
      try
         TLog( LOGID_SYSTEM, TrGl (kLog, 'System.Start', 'Start' ) );

         // Check for SMTP-Server
         if CfgHamster.SmtpServerCount < 1 then begin
            TLog( LOGID_INFO, TrGl(kLog, 'Info.NoSMTPServerDefined', 'No SMTP-server defined.') );
            exit;
         end;
         LfdServer := SearchSMTPServer;
         if LfdServer<0 then begin
            TLog( LOGID_WARN, TrGl(kLog, 'Warning.MailNotSend.NoSMTPServSel',
               'Mail not sent: No SMTP-server selected.' ) );
            exit;
         end;

         // Creating Send-Thread...
         ThreadName := '{smtp to ' + CfgHamster.SmtpServerName[LfdServer] + '}';
         TLog( LOGID_INFO, TrGl(kLog, 'Info.Connecting', 'Connecting ...' ) );
         {MG}{SSL}
         try
            With CfgHamster do begin
               SendMailClient := TClientSMTP.Create(
                  SmtpServerName[LfdServer], SmtpServerPort[LfdServer],
                  SmtpServerUser[LfdServer], SmtpServerPass[LfdServer],
                  SmtpServerMode[LfdServer],
                  SmtpServerVerify[LfdServer], SmtpServerCaFile[LfdServer])
            end;
            try
               SendMailClient.ScriptAuth:=False;
               SendMailClient.Connect;
               repeat
                  if Terminated then break;
                  If ShutdownReq then break;
                  If SendMailClient.State in [csERROR, csDISCONNECTED] then break;
                  TLog( LOGID_INFO, TrGlF(kLog, 'Info.SMTP.Sending','Send mail "%s"', SR.Name));
                  if not SendMailClient.SendMailfile( PATH_MAIL_OUT + SR.Name ) then break;
               until FindNext( SR ) <> 0;
               TLog( LOGID_INFO, TrGl(kLog, 'Info.Disconnecting', 'Disconnecting ...' ) );
               try SendMailClient.Disconnect except end;
            finally
               SendMailClient.Free
            end;
            IncCounter( CntOutboxChk, 1 )
         finally
            ThreadName := Threadbez
         end
      finally
         FindClose( SR )
      end
   end else begin
      TLog( LOGID_DEBUG, TrGl(kLog, 'Debug.SMTP.NoMailsToSend', 'No mail to send.' ) )
   end
end;

procedure TThreadPop3AndSmtp.Execute;
var  LfdServer: Integer;
     OK       : Boolean;
     s        : String;
begin
   If Killme then Exit;
   wait; // Tasks!
   TLog( LOGID_SYSTEM, TrGl(kLog, 'System.Start', 'Start' ));
   // get mails from selected servers
   If Mode IN [0,1] then begin
      For LfdServer:=0 to CfgHamster.Pop3ServerCount-1 do begin
         if Terminated then break;
         if ShutdownReq then break;
         if ServerList='' then begin
            OK := True;
         end else begin
            s := CfgHamster.Pop3ServerName[LfdServer] + ',' + CfgHamster.Pop3ServerPort[LfdServer];
            ok := Pos( ';' + LowerCase(s) + ';', ';' + LowerCase(ServerList) + ';' ) > 0
         end;
         if OK then begin
            ThreadName := '{pop3 from ' + CfgHamster.Pop3ServerName[LfdServer] + '}';
            MailFromServer( LfdServer );
            ThreadName := Threadbez;
         end;
      end
   end;
   If Mode IN [0,2] then begin
      EnterCriticalSection( CS_LOCK_MAILOUT_USE );
      EnterCriticalSection( CS_LOCK_MAILOUT_ADD );
      try
         SendMail;
      finally
         LeaveCriticalSection( CS_LOCK_MAILOUT_ADD );
         LeaveCriticalSection( CS_LOCK_MAILOUT_USE );
      end
   end;
   MainLog( LOGID_SYSTEM, TrGl(kLog, 'System.End', 'End' ));
end;

constructor TThreadPop3AndSmtp.Create( Const AServerList: String; Const AMode: Integer );
Var i, p: Integer; s: String;
begin
   KillMe := false;
   Mode := AMode;
   // Check serverlist
   ServerList := AServerList;
   Mode := AMode;
   if ServerList > '' then begin
      With TStringList.Create do try
         Text := SplitServerList (ServerList);
         ServerList := '';
         ThreadBez := '';
         For i := 0 to Count-1 do begin
            s := Strings[i];
            If ((CfgHamster.POP3ServerIndexof[s] >= 0) and (Mode IN[0,1]) )
                or ((CfgHamster.SMTPServerIndexof[s] >= 0) and (Mode IN[0,2]) )
            then begin
               ServerList := ServerList + ';' + s;
               p := Pos(',', s);
               If p > 0 then s := Copy(s, 1, p-1);
               If ThreadBez > '' then ThreadBez := ThreadBez + ', ';
               ThreadBez := ThreadBez + s
            end else begin
               Log ( LOGID_WARN, TrGlF(kLog, 'MailExchange.UnknownServer',
                  'MailExchange: Unknown server "%s"!', s))
            end
         end;
         If ServerList = ''
            then KillMe := true
            else ServerList := ServerList + ';'
      finally
         free
      end
   end else begin
      ThreadBez := '*'
   end;
   inherited Create( '{pop3/smtp '+ThreadBez+'}' );
end;

// ---------------------------------------------------------- TThreadPull -----

procedure TThreadNewsJobs.NukePostedArticle( Const PostFile, ArtText, DestGroup, Reason,
   DestSrv: String; Const PostedOK: Boolean );
var  OldGroup, MessageID, s, SaveText: String; DestHdl, ArtNo: Integer;
begin
   // delete posted article, even if post failed
   DeleteFile( PostFile );

   // prepare article to post a copy of it to internal.misc
   With TArticle.Create do try
      Text  := ArtText;
      s := Header['Path:'];
      if s = '' then begin
         if Def_FQDN <>''
            then AddHeaderFirst('Path:', Def_FQDN+'!not-for-mail' )
            else AddHeaderFirst('Path:', 'not-for-mail' )
      end;
      OldGroup := Header['Newsgroups'];
      RenameHeader('Newsgroups', 'X-Post-Newsgroups');
      AddHeaderFirst( 'Newsgroups:', DestGroup );
      MessageID := Header['Message-ID:'];
      if MessageID = ''
         then MessageID := MidGenerator( 'posted.without.mid.invalid' )
         else RenameHeader('Message-ID', 'X-Post-Message-ID');
      If PostedOK
         then Insert('archive.', MessageID, 2)
         else Insert('failed.', MessageID, 2);
      AddHeaderFirst('Message-ID:', MessageID );
      If Header['Date:'] = '' then begin
         AddHeaderFirst( 'Date:', DateTimeGMTToRfcDateTime(NowGMT,NowRfcTimezone) )
      end;
      If PostedOK then s := 'posted' else s := 'rejected';
      FullBody :=  '[Hamster: archive-copy of '+s+' article ('+ OldGroup + ')]' + #13#10
             + 'Result: ' + Reason + #13#10 + #13#10
             + FullBody;
      Header['Lines:'] := IntToStr(CountLines(FullBody));
      SaveText := Text
   finally
      Free
   end;
   // add timestamp and post-result
   s := 'Received=' + DateTimeToTimeStamp( Now ) + ' '
      + 'Server=' + DestSrv + ' '
      + 'Result=' + ReplaceChar( Reason, ' ', '_' );
   SaveText := OUR_X_HEADER + ' ' + s + #13#10 + SaveText;
   // save in DestGroup
   DestHdl := ArticleBase.Open( DestGroup );
   if DestHdl>=0 then begin
      ArtNo := ArticleBase.WriteArticle( DestHdl, 0, SaveText );
      ArticleBase.Close( DestHdl );
      NewsHistory.AddEntryDupes( MessageID, StrToCRC32(LowerCase(DestGroup)), ArtNo, 0 );
   end else begin
      TLog( LOGID_ERROR, TrGlF(kLog, 'Error.CouldntOpenGroupToArchivePostedArticle',
            'Could not open %s to archive posted article', DestGroup))
   end
end;

procedure TThreadNewsJobs.PostArticle( PostFile: String );
var  Art       : TStringlist;
     SrvResult : String;
     DestGroup : String;
     Body      : String; //JW //NNTPINFOMAIL
     bOK       : Boolean;
begin
   if FileExists2( PostFile ) then begin
      Art := TStringlist.Create;
      try
         Art.LoadFromFile( PostFile );
         EnterCriticalSection( CS_LOCK_NEWSOUT );
         try
            bOK := NewsClient.PostArticle( PostFile, Art.Text, SrvResult );
            If bOK then begin
               DestGroup := INTERNALGROUP[INTERNALGROUP_POSTOK];
               TLog( LOGID_INFO, TrGlF (kLog, 'Info.Posting.ok',
                  'Posting OK: %s (%s)', [SrvResult,ExtractFileName(PostFile)]));
            end else begin
               DestGroup := INTERNALGROUP[INTERNALGROUP_POSTERRORS];
               With TFileStream.Create( PATH_NEWS_ERR + ExtractFileName(Postfile), fmCreate) do try
                  Write ( Art.Text[1], Length(Art.Text) )
               finally free end;
               TLog( LOGID_WARN, TrGlF(kLog, 'Warning.Posting.failed',
                  'Posting failed: %s (%s)', [SrvResult,ExtractFileName(PostFile)]));
               If Def_NNTPINFOMAIL then begin
                 Body:='[Usenet]'+CRLF+
                       'Posting failed, Server result: '+SrvResult+CRLF+
                       'stored in :'+PATH_NEWS_ERR+ExtractFileName(Postfile);
                 SendInfoMail(Def_Usenet,'Usenet: posting failed', '', body);
               end;
            end;
            // save article in DestGroup and delete it from Post-group/-list
            NukePostedArticle( PostFile, Art.Text, DestGroup, SrvResult,
                               CfgHamster.ServerName[LfdServer], bOK );
            IncCounter( CntOutboxChk, 1 );
         finally
            LeaveCriticalSection( CS_LOCK_NEWSOUT );
         end
      finally
         Art.Free;
      end
   end
end;

procedure TThreadNewsJobs.Execute;
var  ServerName, GroupName, JobPar   : String;
     JobType: TJobtype;
     GroupHandle, DelayCount: Integer;
     IsConnected, OK                 : Boolean;
begin
   // Make sure, a specific server is not used too often by concurrent threads
   if not CfgHamster.ServerUseInc( LfdServer ) then begin
      // the jobs for this server are already executed by other threads
      Terminate;
      exit;
   end;
   wait; {JW} {Tasks}

   TLog( LOGID_SYSTEM, TrGl(kLog, 'System.Start', 'Start' ) );
   ServerName := CfgHamster.ServerName[ LfdServer ];

   NewsClient  := nil;
   IsConnected := False;
   DelayCount  := 0;

   repeat
      // stop download?
      if ShutdownReq or Terminated then break;

      // get (next) job
      if not NewsJobs.JobList.JobGet( ServerName, JobType, JobPar ) then break;

      // connect to server
      if not IsConnected then begin
         TLog( LOGID_INFO, TrGl(kLog, 'Info.Connecting', 'Connecting ...' ) );
         {MG}{SSL}
         NewsClient := TClientNNTP.Create( CfgHamster.ServerName[LfdServer],
                                           CfgHamster.ServerPort[LfdServer],
                                           CfgHamster.ServerUser[LfdServer],
                                           CfgHamster.ServerPass[LfdServer],
                                           CfgHamster.ServerMode[LfdServer],
                                           CfgHamster.ServerVerify[LfdServer],
                                           CfgHamster.ServerCaFile[LfdServer],
                                           CfgHamster.NNTPServertype[LfdServer]
                                           );
         {/SSL}
         NewsClient.Connect(Jobtype<>JOBTYPE_NEWSFEED); {JW} {Mode Reader}
         if NewsClient.State in [csERROR, csDISCONNECTED] then break;
         IsConnected := True;
      end;

      // execute job
      case JobType of

         JOBTYPE_SRVINFOS: begin
            // get infos for new servers and new groups on old servers
            TLog( LOGID_INFO, TrGl(kLog, 'Infos.GetServerInfos', 'get server-info' ) );
            NewsClient.GetServerInfos;
         end;

         JOBTYPE_GETBYMID: begin
            // get articles by list of Message-IDs
            TLog( LOGID_INFO, TrGl(kLog, 'Info.GetListOfMIDs', 'get list of MIDs' ) );
            NewsClient.GetListOfMIDs;
         end;

         JOBTYPE_NEWSPOST: begin
            TLog( LOGID_INFO, TrGlF(kLog, 'Info.PostArticle', 'post %s', ExtractFilename(JobPar)) );
            PostArticle( JobPar ); // post article
         end;
{JW} {Feed}
         JOBTYPE_NEWSFEED: begin
            //  feeding articles for group to remote server
            GroupName := JobPar;
            TLog( LOGID_INFO, TrGlF(kLog, 'Info.FeedGroup', 'feeding %s', GroupName) );
            GroupHandle := ArticleBase.Open( GroupName );
            if GroupHandle>=0 then begin
               try
                  NewsClient.FeedNewsForGroup(GroupName, GroupHandle );
               finally
                  ArticleBase.Close( GroupHandle );
               end;
            end;
         end;
{JW}
         JOBTYPE_NEWSPULL: begin
            // load new articles for group
            GroupName := JobPar;
            TLog( LOGID_INFO, TrGlF(kLog, 'Info.PullGroup', 'pull %s', GroupName) );

            OK := True;
            if ArticleBase.UseCount( GroupName ) > 0 then begin
               // Avoid ineffective downloads of groups, that are already "in
               // use" by other (pull-) threads. Due to synchronized access on
               // group files, they would block each other and delay download.
               OK := False;
               inc( DelayCount );
               if DelayCount>=3 then begin
                  OK := True;
                  DelayCount := 0;
               end;
               if not OK then begin
                  TLog( LOGID_DEBUG, TrGlF(kLog, 'Debug.GroupInUseTryingLater',
                     'Group %s already in use, trying later.', GroupName));
                  Sleep( 500 );
                  // Put job back into job-list as (up to) 4th job for server.
                  NewsJobs.JobList.JobDelay( ServerName, JobType, JobPar, 3 );
               end;
            end;

            if OK then begin
               GroupHandle := ArticleBase.Open( GroupName );
               if GroupHandle>=0 then begin
                  NewsClient.GetNewArticlesForGroup( GroupName, GroupHandle );
                  ArticleBase.Close( GroupHandle );
               end;
            end;
         end;
         else TLog( LOGID_WARN, TrGl(kLog, 'Warning.UnKnownJobtype',
            'Unknown job-type') + ': ' + inttostr(Ord(JobType)) );
      end;

   until NewsClient.State in [csERROR, csDISCONNECTED];


   if Assigned( NewsClient ) then begin
      // disconnect from server
      TLog( LOGID_INFO, TrGl(kLog, 'Info.Disconnecting', 'Disconnecting ...' ) );
      if IsConnected then try NewsClient.Disconnect; except end;
      try NewsClient.Free; except end;
   end else begin
      TLog( LOGID_INFO, TrGlF(kLog, 'Info.NewsjobsAbandonedNoJobs',
         'News-jobs for %s abandoned (no jobs)', ServerName) );
   end;

   {AS} { On the termination of the last thread for a given server,
		  drain any residual job in the list (after a connection error, etc.) }
   if CfgHamster.ServerUseDec( LfdServer ) = 0 then begin
      If Def_NNTPDropResidualJobs then begin
         while NewsJobs.JobList.JobGet(ServerName, JobType, JobPar) do begin
             TLog( LOGID_WARN, TrGlF(kLog, 'Info.NewsFlushingJob',
                'Drop residual job: %s', JobInfo(JobType, JobPar)) )
         end
      end
   end;
   {/AS}
   NewsHistory.SaveToFile;

   TLog( LOGID_SYSTEM, TrGl(kLog, 'System.End', 'End' ) );
end;

{JW} {ServerStart}
constructor TThreadNewsJobs.Create( Const ALfdServer: Integer );
begin
   LfdServer := ALfdServer;
   inherited Create( '{newsjobs ' + CfgHamster.ServerName[LfdServer] + '}' );
   NewsClient := nil;
end;

destructor TThreadNewsJobs.destroy;
begin
   Mainlog(LOGID_SYSTEM, TrGlF(kLog, 'newsjobs.end', 'newsjobs for %s ready', CfgHamster.ServerName[LfdServer] ) );
   inherited;
end;

// ---------------------------------------------------------- TThreadPost -----

procedure TThreadNewsPostSimulate.Execute;

   function IsServerSelected( TestSrv, TestPort: String ): Boolean;
   begin
      if ServerList='' then begin
         Result := True;
      end else begin
         Result := Pos(';'+LowerCase(TestSrv+','+TestPort)+';',
                       ';'+LowerCase(ServerList)+';') >0;
      end;
   end;

var  PostServer: String;
     PostSrvIdx: Integer;
begin
   TLog( LOGID_SYSTEM, TrGl(kLog, 'System.Start', 'Start' ) );
   // add messages to job-list
   NewsJobs.AddPostDef( ServerList );
   // start a job-thread for all selected servers, which have post-jobs
   for PostSrvIdx:=0 to CfgHamster.ServerCount-1 do begin
      PostServer := CfgHamster.ServerName[PostSrvIdx];
      if IsServerSelected( PostServer, CfgHamster.ServerPort[PostSrvIdx] ) then begin
         if NewsJobs.JobList.JobIndexST(PostServer, JOBTYPE_NEWSPOST)>=0 then begin
            TThreadNewsJobs.Create( PostSrvIdx).resume;
         end
      end
   end;
   TLog( LOGID_SYSTEM, TrGl(kLog, 'System.End', 'End' ) );
end;

constructor TThreadNewsPostSimulate.Create( Const AServerList: String );
begin
   inherited Create( '{NewsPost-macro}' );
   ServerList := AServerList;
   if ServerList<>'' then ServerList := ';' + LowerCase(ServerList) + ';';
end;

destructor TThreadNewsPostSimulate.destroy;
begin
   Mainlog(LOGID_SYSTEM, TrGl(kLog, 'newspostsim.end', 'NewsPost-macro ready') );
   inherited;
end;

end.
