// ============================================================================
// Dynamically loaded RAS-functions
// Copyright (c) 1999, Juergen Haible. All Rights Reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit uRasDyn; // Dynamically loaded RAS-functions

// ----------------------------------------------------------------------------
// Contains a subset of RAS-API and some helper functions. All RAS-functions
// are loaded dynamically when needed/requested, so it's not necessary that
// RAS has to be installed for starting and using the program.
// ----------------------------------------------------------------------------

interface

uses SysUtils, Windows, Classes, tBase;

const
  RASCS_PAUSED = $1000;
  RASCS_DONE   = $2000;

  RASCS_OpenPort = 0;
  RASCS_PortOpened = 1;
  RASCS_ConnectDevice = 2;
  RASCS_DeviceConnected = 3;
  RASCS_AllDevicesConnected = 4;
  RASCS_Authenticate = 5;
  RASCS_AuthNotify = 6;
  RASCS_AuthRetry = 7;
  RASCS_AuthCallback = 8;
  RASCS_AuthChangePassword = 9;
  RASCS_AuthProject = 10;
  RASCS_AuthLinkSpeed = 11;
  RASCS_AuthAck = 12;
  RASCS_ReAuthenticate = 13;
  RASCS_Authenticated = 14;
  RASCS_PrepareForCallback = 15;
  RASCS_WaitForModemReset = 16;
  RASCS_WaitForCallback = 17;
  RASCS_Projected = 18;
{$IFNDEF WINVER31}
  RASCS_StartAuthentication = 19;
  RASCS_CallbackComplete = 20;
  RASCS_LogonNetwork = 21;
{$ENDIF}
  RASCS_Interactive = RASCS_PAUSED;
  RASCS_RetryAuthentication = RASCS_PAUSED + 1;
  RASCS_CallbackSetByCaller = RASCS_PAUSED + 2;
  RASCS_PasswordExpired = RASCS_PAUSED + 3;
  RASCS_Connected = RASCS_DONE;
  RASCS_Disconnected = RASCS_DONE + 1;

  DNLEN = 15;  // Maximum domain name length
  UNLEN = 256; // Maximum user name length
  PWLEN = 256; // Maximum password length

  RAS_MaxDeviceType  = 16;
  RAS_MaxPhoneNumber = 128;
  RAS_MaxIpAddress   = 15;
  RAS_MaxIpxAddress  = 21;
  RAS_MaxEntryName      = 256;
  RAS_MaxDeviceName     = 128;
  RAS_MaxCallbackNumber = RAS_MaxPhoneNumber;

  RASP_PppIp = $8021;

type
  THRasConn = Longint;
  TRasConnState = Integer;

  TRasDialParamsA = record
    dwSize: LongInt;
    szEntryName: Array[0..RAS_MaxEntryName] of AnsiChar;
    szPhoneNumber: Array[0..RAS_MaxPhoneNumber] of AnsiChar;
    szCallbackNumber: Array[0..RAS_MaxCallbackNumber] of AnsiChar;
    szUserName: Array[0..UNLEN] of AnsiChar;
    szPassword: Array[0..PWLEN] of AnsiChar;
    szDomain: Array[0..DNLEN] of AnsiChar;
  end;
  TRasDialParams = TRasDialParamsA;

  TRasConnStatusA = record
    dwSize: Longint;
    rasconnstate: TRasConnState;
    dwError: LongInt;
    szDeviceType: Array[0..RAS_MaxDeviceType] of AnsiChar;
    szDeviceName: Array[0..RAS_MaxDeviceName] of AnsiChar;
  end;
  TRasConnStatus = TRasConnStatusA;

  LPRasConnA = ^TRasConnA;
  TRasConnA = record
    dwSize: Longint;
    hrasconn: THRasConn;
    szEntryName: Array[0..RAS_MaxEntryName] of AnsiChar;
    szDeviceType: Array[0..RAS_MaxDeviceType] of AnsiChar;
    szDeviceName: Array[0..RAS_MaxDeviceName] of AnsiChar;
  end;

  LPRasEntryNameA = ^TRasEntryNameA;
  TRasEntryNameA = record
    dwSize: Longint;
    szEntryName: Array[0..RAS_MaxEntryName] of AnsiChar;
  end;
  TRasEntryName  = TRasEntryNameA;

  LPRasDialExtensions = ^TRasDialExtensions;
  TRasDialExtensions = record
    dwSize: LongInt;
    dwfOptions: LongInt;
    hwndParent: HWND;
    reserved: LongInt;
  end;

type
  TRasDialA             = function (
                            lpRasDialExt: LPRasDialExtensions;
                            lpszPhoneBook: PAnsiChar;
                            var params: TRasDialParamsA;
                            dwNotifierType: Longint;
                            lpNotifier: Pointer;
                            var rasconn: THRasConn
                          ): Longint; stdcall;

  TRasHangUpA           = function (
                            hConn: THRasConn
                          ): Longint; stdcall;

  TRasGetConnectStatusA = function (
                            hConn: THRasConn;
                            var lpStatus: TRasConnStatusA
                          ): Longint; stdcall;

  TRasEnumConnectionsA =  function (
                            rasconnArray: LPRasConnA;
                            var lpcb: Longint;
                            var lpcConnections: Longint
                          ): Longint; stdcall;

  TRasEnumEntriesA =      function (
                            reserved: PAnsiChar;
                            lpszPhoneBook: PAnsiChar;
                            entrynamesArray: LPRasEntryNameA;
                            var lpcb: Longint;
                            var lpcEntries: Longint
                          ): Longint; stdcall;

  TRasGetErrorStringA =   function (
                            errorValue: Integer;
                            erroString: PAnsiChar;
                            cBufSize: Longint
                          ): Longint; stdcall;

  TRasProjection = Integer;

  TRasGetProjectionInfoA= function (
                            hConn: THRasConn;
                            rasproj: TRasProjection;
                            lpProjection: Pointer;
                            var lpcb: Longint
                          ): Longint; stdcall;

var
  RasDynDll: HModule = 0;

  RasDynDial            : TRasDialA              = nil;
  RasDynHangup          : TRasHangUpA            = nil;
  RasDynGetConnectStatus: TRasGetConnectStatusA  = nil;
  RasDynEnumConnections : TRasEnumConnectionsA   = nil;
  RasDynEnumEntries     : TRasEnumEntriesA       = nil;
  RasDynGetErrorString  : TRasGetErrorStringA    = nil;
  RasGetProjectionInfo  : TRasGetProjectionInfoA = nil;

function  RasDynErrTxt( ErrNo: LongInt ): String;
function  RasDynGetPppIp( hConn: THRasConn ): String;

function  RasDynInit: Boolean;
procedure RasDynExit;
function  RasDynProcAddress( ProcName: String ): Pointer;

function  RasDynIsConnected: Boolean;
function  RasDynGetConnection: String;
function  RasDynGetConn: Integer;
function  RasDynEnumPhonebookEntries( RasList: TStringList ): Boolean;

Type
  TThreadRASDial = class( TTaskThread )
  protected
    fRASName: String;
    procedure Execute; override;
  public
    constructor Create (Const RASName: String);
  end;

// ----------------------------------------------------------------------------

implementation

Uses Config;

function GetCurrentRasConnection( var ConnectionID: String ): LongInt;
const RasConnMax = 3;
var  RasConn : array[0..RasConnMax] of TRasConnA;
     RasConnSize : LongInt;
     RasConnCount: LongInt;
     RasConnStat : TRasConnStatusA;
     ConnectionState: Integer;
begin
     // default: no connection
     Result := 0;
     ConnectionID := '';
     if not RasDynInit then exit;

     try
        // enum active connections
        RasConnSize := SizeOf( RasConn );
        RasConn[0].dwSize := SizeOf( TRasConnA );
        if RasDynEnumConnections(@RasConn[0],RasConnSize,RasConnCount)<>0 then exit;
        if RasConnCount < 1 then exit;

        // get connection-state of first connection
        RasConnStat.dwSize := SizeOf( TRasConnStatus );
        if RasDynGetConnectStatus(RasConn[0].hrasconn,RasConnStat)<>0 then exit;
        ConnectionState := RasConnStat.rasconnstate;

        // if connected completely, return connection-info
        if ConnectionState=RASCS_Connected then begin
           ConnectionID := RasConn[0].szEntryName;
           Result       := RasConn[0].hrasconn;
        end;
     except
        // ignore
     end;
end;

function RasDynIsConnected: Boolean;
var  i: Integer;
     s: String;
begin
   i := GetCurrentRasConnection( s );
   if (i<>0) or (s<>'') then Result:=True else Result:=False;
end;

function RasDynGetConnection: String;
begin
   GetCurrentRasConnection( Result );
end;

function RasDynGetConn: Integer;
var  s: String;
begin
   Result := GetCurrentRasConnection( s );
end;

function RasDynEnumPhonebookEntries( RasList: TStringList ): Boolean;
var  RasEntries     : array of TRasEntryName;
     RasEntriesSize : LongInt;
     RasEntryCount  : LongInt;
     l              : LongInt;
begin
   Result := False;
   RasList.Clear;
   if not RasDynInit then exit;

   Result := True;
            // 792
   SetLength(RasEntries, 1);
   RasEntries[0].dwSize := SizeOf(TRasEntryName);
   RasEntriesSize := SizeOf( TRasEntryName );
   RasDynEnumEntries( PChar(0), PChar(0), @RasEntries[0],
                         RasEntriesSize, RasEntryCount ); // = 0
   If RasEntriesSize > SizeOf(TRasEntryName) then begin
      SetLength(RasEntries, RasEntriesSize div SizeOf(TRasEntryName))
   end;
   if RasDynEnumEntries( PChar(0), PChar(0), @RasEntries[0],
                         RasEntriesSize, RasEntryCount ) = 0
   then begin
      for l:=0 to RasEntryCount-1 do begin
         RasList.Add( RasEntries[l].szEntryName );
      end;
   end;
end;

function RasDynInit: Boolean;
begin
     Result := False;

     if RasDynDll=0 then begin
        try
           RasDynDll := SafeLoadLibrary( 'rasapi32.dll', SEM_FAILCRITICALERRORS );
           if RasDynDll=0 then exit;
           @RasDynDial             := GetProcAddress( RasDynDll, 'RasDialA' );
           @RasDynHangup           := GetProcAddress( RasDynDll, 'RasHangUpA' );
           @RasDynGetConnectStatus := GetProcAddress( RasDynDll, 'RasGetConnectStatusA' );
           @RasDynEnumConnections  := GetProcAddress( RasDynDll, 'RasEnumConnectionsA' );
           @RasDynEnumEntries      := GetProcAddress( RasDynDll, 'RasEnumEntriesA' );
           @RasDynGetErrorString   := GetProcAddress( RasDynDll, 'RasGetErrorStringA' );
           @RasGetProjectionInfo   := GetProcAddress( RasDynDll, 'RasGetProjectionInfoA' );
           Result := True;
        except
           // on E:Exception do msglog( 'RasDynInit: ' + E.Message );
        end;
     end else begin
        Result := True;
     end;
end;

procedure RasDynExit;
begin
     if RasDynDll<>0 then begin
        FreeLibrary( RasDynDll );
        RasDynDll := 0;
     end;
end;

function RasDynProcAddress( ProcName: String ): Pointer;
begin
     Result := nil;
     if not RasDynInit then exit;
     Result := GetProcAddress( RasDynDll, PChar( ProcName ) );
end;

function RasDynErrTxt( ErrNo: LongInt ): String;
var  pc: array[0..256] of char;
begin
     if RasDynGetErrorString( ErrNo, pc, 256 )=0 then Result:=String(pc)
                                                 else Result:='';
end;

type
  RASPPPIP = record
    dwSize: LongInt;
    dwError: LongInt;
    szIpAddress: array [0..RAS_MaxIpAddress] of AnsiChar;
  end;

function RasDynGetPppIp( hConn: THRasConn ): String;
var  lpcb: Longint;
     Projection: RASPPPIP;
begin
     Result := '';
     if not Assigned( RasGetProjectionInfo ) then exit;

     Projection.dwSize := sizeof( Projection );
     if RasGetProjectionInfo( hConn, RASP_PppIp, @Projection, lpcb )=0 then begin
        if Projection.dwError=0 then begin
           Result := String( Projection.szIpAddress );
        end;
     end;
end;

// ----------------------------------------------------------------------------

{ TThreadRASDial }

constructor TThreadRASDial.Create (Const RASName: String);
begin
   inherited Create( '{RAS-Dial}' );
   fRASName := RASName;
end;

procedure TThreadRASDial.Execute;
begin
  If Not Assigned(RasDialer) then Rasdialer := TRasDialer.Create;
  RasDialer.Dial ( fRASName, '', '' )
end;

initialization

finalization
  RasDynExit;

end.

