// ============================================================================
// Hamster, a free news- and mailserver for personal, family and workgroup use.
// Copyright (c) 1999, Juergen Haible.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit uSSL; // Interface to the OpenSSL cryptography library

// ----------------------------------------------------------------------------
// For details about OpenSSL, see http://www.openssl.org/
// Unit created by Martin Germann <martin.germann@gmx.net>
// ----------------------------------------------------------------------------

interface

uses Classes;

type
   TSSLContext = class( TObject )
      private
         FContext: Pointer;
         FVerify, FOptions: Integer;
         FCaFile, FCaPath: PChar;
      public
         procedure SetOptions( CipherList: String ); {MG}{CipherList}
         function LoadPrivateKeyPair( AKey: PChar ): Boolean;
         property Context: Pointer read FContext;
         constructor Create( AServer: Boolean; AVerify: Integer;
                             ACaFile, ACaPath: String );
         destructor Destroy; override;
   end;

   TSSLConnection = class( TObject )
      private
         FSsl    : Pointer;
      public
         function Connect( const ASocketHandle: Integer ): Boolean;
         function Accept( const ASocketHandle: Integer ): Boolean;
         procedure Info;
         function HasPendingData: Boolean;
         function Read( Buf: PChar; Num: Integer): Integer;
         function Write( Buf: PChar; Num: Integer): Integer;
         procedure Shutdown;
         property Ssl: Pointer read FSsl;
         constructor Create( ASSLContext: Pointer);
         destructor Destroy; override;
   end;

function InitializeSSL : Boolean;
procedure FreeSSL;

Procedure FillSSLModeList (List: TStrings);
Procedure FillSSLVerifyModeList (List: TStrings);

function OpenSSLVersion: Integer; {MG}{AES}

function SSL_RandBytes( out Buf: Array of Char; Bytes: Integer ): Boolean;

// ----------------------------------------------------------------------------

implementation

uses Windows, SysUtils, Global, Config, cStdForm, cLogFile, cPasswordFile;

const
   // Maximum of random bytes to read from random seed file
   MAX_RANDOM_BYTES = 256;

   // OpenSSL 0.9.6b or higher is required
   REQUIRED_OPENSSL_VERSION = $0090602f;

   // Used to store the password for the private key in the password file
   KEYID = 'OpenSSL private certificate';

   // Constants from the OpenSSL header files
   SSLEAY_VERSION  = 0;
   SSL_VERIFY_NONE = 0;
   SSL_VERIFY_PEER = 1;
   SSL_VERIFY_FAIL_IF_NO_PEER_CERT = 2;
   SSL_VERIFY_CLIENT_ONCE = 4;
   SSL_VERIFY_ONLY_LOCAL_CERTS = 8;
   SSL_SENT_SHUTDOWN = 1;
   SSL_RECEIVED_SHUTDOWN = 2;
   SSL_CTRL_OPTIONS = 32;
   SSL_CTRL_MODE = 33;
   SSL_OP_NO_SSLv2 = $01000000;
   SSL_OP_NO_SSLv3 = $02000000;
   SSL_OP_NO_TLSv1 = $04000000;
   SSL_OP_ALL = $000FFFFF;  // all bug workaround options
   SSL_MODE_ENABLE_PARTIAL_WRITE = $00000001;
   SSL_MODE_AUTO_RETRY = $00000004;
   SSL_ERROR_NONE = 0;
   SSL_ERROR_SSL = 1;
   SSL_ERROR_SYSCALL = 5;
   SSL_ERROR_ZERO_RETURN = 6;

var
   LibeayHandle : THandle;
   LibsslHandle : THandle;

   OpenSSL_SSL_load_error_strings: procedure cdecl = nil;
   OpenSSL_SSL_library_init: function: Integer cdecl = nil;
   OpenSSL_SSLeay: function: LongInt cdecl = nil;
   OpenSSL_SSLeay_version: function(t : Integer): PChar cdecl = nil;

   OpenSSL_RAND_load_file: function(const PFilename: PChar; max_bytes: LongInt): Integer cdecl = nil;
   OpenSSL_RAND_write_file: function(const PFilename: PChar): Integer cdecl = nil;
   OpenSSL_RAND_screen: procedure cdecl = nil;
   OpenSSL_RAND_status: function: Integer cdecl = nil;
   OpenSSL_RAND_bytes: function(buf: PChar; num: Integer): Integer cdecl = nil; {PRNG}

   OpenSSL_SSLv23_client_method: function: Pointer cdecl = nil;
   OpenSSL_SSLv23_server_method: function: Pointer cdecl = nil;

   OpenSSL_SSL_CTX_new: function(method: Pointer): Pointer cdecl = nil;
   OpenSSL_SSL_CTX_free: procedure(ctx: Pointer) cdecl = nil;
   OpenSSL_SSL_CTX_ctrl: function(ctx: Pointer; cmd: Integer; larg: LongInt; parg: PChar ): LongInt cdecl = nil;
   OpenSSL_SSL_CTX_set_default_passwd_cb: procedure(ctx: Pointer; callback: Pointer) cdecl = nil;
   OpenSSL_SSL_CTX_set_default_passwd_cb_userdata: procedure(ctx, u: Pointer) cdecl = nil;
   OpenSSL_SSL_CTX_set_verify: procedure(ctx: Pointer; mode: Integer; verify_callback: Pointer) cdecl = nil;
   OpenSSL_SSL_CTX_set_cipher_list: function(ctx: Pointer; str: PChar): Integer cdecl = nil;
   OpenSSL_SSL_CTX_use_PrivateKey_file: function(ctx: Pointer; const _file: PChar; _type: Integer): Integer cdecl = nil;
   OpenSSL_SSL_CTX_load_verify_locations: function(ctx: Pointer; CAfile, CApath: PChar): Integer cdecl = nil;
   OpenSSL_SSL_CTX_check_private_key: function(ctx: Pointer): Integer cdecl = nil;
   OpenSSL_SSL_CTX_use_certificate_file: function(ctx: Pointer; const _file: PChar; _type: Integer): Integer cdecl = nil;

   OpenSSL_SSL_new: function(ctx: Pointer): Pointer cdecl = nil;
   OpenSSL_SSL_free: procedure(ssl: Pointer) cdecl = nil;
   OpenSSL_SSL_accept: function(ssl: Pointer): Integer cdecl = nil;
   OpenSSL_SSL_connect: function(ssl: Pointer): Integer cdecl = nil;
   OpenSSL_SSL_pending: function(ssl: Pointer): Integer cdecl = nil;
   OpenSSL_SSL_read: function(ssl: Pointer; buf: PChar; num: Integer): Integer cdecl = nil;
   OpenSSL_SSL_write: function(ssl: Pointer; const buf: PChar; num: Integer): Integer cdecl = nil;
   OpenSSL_SSL_shutdown: function(ssl: Pointer): Integer cdecl = nil;
   OpenSSL_SSL_set_fd: function(ssl: Pointer; fd: Integer): Integer cdecl = nil;
   OpenSSL_SSL_set_shutdown: procedure(ssl: Pointer; mode: Integer) cdecl = nil;
   OpenSSL_SSL_get_version: function(ssl: Pointer): PChar cdecl = nil;
   OpenSSL_SSL_get_current_cipher: function(ssl: Pointer): Pointer cdecl = nil;
   OpenSSL_SSL_get_error: function(ssl: Pointer; ret: Integer): Integer cdecl = nil;
   OpenSSL_X509_get_subject_name: function(x509: Pointer): Pointer cdecl = nil;
   OpenSSL_X509_NAME_oneline: function(x509_name: Pointer; buf: PChar; size: Integer): PChar cdecl = nil;
   OpenSSL_X509_verify_cert_error_string: function(n: Longint): PChar cdecl = nil;
   OpenSSL_X509_STORE_CTX_get_current_cert: function(x509_store_ctx: Pointer): Pointer cdecl = nil;
   OpenSSL_X509_STORE_CTX_get_error: function(x509_store_ctx: Pointer): Integer cdecl = nil;
   OpenSSL_X509_STORE_CTX_get_error_depth: function(x509_store_ctx: Pointer): Integer cdecl = nil;
   OpenSSL_X509_STORE_get_by_subject: function(x509_store_ctx: Pointer; _type: Integer; name, ret: Pointer): Integer cdecl = nil;

   OpenSSL_ERR_get_error: function: LongInt cdecl = nil;
   OpenSSL_ERR_error_string: function(err: LongInt; pc: PChar): PChar cdecl = nil;
   OpenSSL_ERR_remove_state: procedure(pid: LongInt) cdecl = nil;

   OpenSSL_SSL_CIPHER_get_bits: function(cipher, alg_bits: Pointer): Integer cdecl = nil;
   OpenSSL_SSL_CIPHER_get_name: function(cipher: Pointer): PChar cdecl = nil;
   OpenSSL_SSL_CIPHER_description: function(cipher: Pointer; buf: PChar; size: Integer): PChar cdecl = nil;


procedure GetError( txt: String );
var  err   : LongInt;
     buf   : Array[0..120] of char;
     error : String;
begin
     err := OpenSSL_ERR_get_error;
     if err <> 0 then begin
        OpenSSL_ERR_error_string( err, buf );
        error := ': ' + string( buf );
     end else // Error queue is empty
        error := '.';
     Log( LOGID_ERROR, Format(TrGl(kLog, 'SSL.Error',
          'OpenSSL error: %s%s'), [txt, error]) );
end;

function LoadFunction( FunctionName: string ): Pointer;
begin
     Result := GetProcAddress( LibeayHandle, PChar( FunctionName ) );
     if Result = nil then
        Log( LOGID_ERROR, Format(TrGl(kLog, 'SSL.LoadFunctionError',
             'OpenSSL: Could not load function "%s" from %s'),
             [FunctionName, 'libeay32.dll']) );
end;

function LoadSSLFunction( FunctionName: string ): Pointer;
begin
     Result := GetProcAddress( LibsslHandle, PChar( FunctionName ) );
     if Result = nil then
        Log( LOGID_ERROR, Format(TrGl(kLog, 'SSL.LoadFunctionError',
             'OpenSSL: Could not load function "%s" from %s'),
             [FunctionName, 'libssl32.dll']) );
end;

function InitializeSSL : Boolean;
var  Randfile : String;
     Bytes : Integer;
begin
     Result := False;

     Log( LOGID_DETAIL, TrGl(kLog, 'SSL.Init',
          'Initializing OpenSSL cryptography libraries ...') );
     LibeayHandle := LoadLibrary( 'libeay32.dll' );
     if LibeayHandle = 0  then begin
        Log( LOGID_DEBUG, Format(TrGl(kLog, 'SSL.DllNotFound',
             'OpenSSL library %s not found'), ['libeay32.dll']) );
        exit;
     end;
     LibsslHandle := LoadLibrary( 'libssl32.dll' );
     if LibsslHandle = 0  then begin
        Log( LOGID_DEBUG, Format(TrGl(kLog, 'SSL.DllNotFound',
             'OpenSSL library %s not found'), ['libssl32.dll']) );
        FreeLibrary( LibeayHandle );
        exit;
     end;

     @OpenSSL_SSL_load_error_strings := LoadSSLFunction('SSL_load_error_strings');
     @OpenSSL_SSL_library_init := LoadSSLFunction('SSL_library_init');
     @OpenSSL_SSLeay := LoadFunction('SSLeay');
     @OpenSSL_SSLeay_version := LoadFunction('SSLeay_version');

     @OpenSSL_RAND_load_file := LoadFunction('RAND_load_file');
     @OpenSSL_RAND_write_file := LoadFunction('RAND_write_file');
     @OpenSSL_RAND_screen := LoadFunction('RAND_screen');
     @OpenSSL_RAND_status := LoadFunction('RAND_status');
     @OpenSSL_RAND_bytes := LoadFunction('RAND_bytes');  {PRNG}

     @OpenSSL_SSLv23_client_method := LoadSSLFunction('SSLv3_client_method');
     @OpenSSL_SSLv23_server_method := LoadSSLFunction('SSLv23_server_method');

     @OpenSSL_SSL_CTX_new := LoadSSLFunction('SSL_CTX_new');
     @OpenSSL_SSL_CTX_free := LoadSSLFunction('SSL_CTX_free');
     @OpenSSL_SSL_CTX_ctrl := LoadSSLFunction('SSL_CTX_ctrl');
     @OpenSSL_SSL_CTX_set_verify := LoadSSLFunction('SSL_CTX_set_verify');
     @OpenSSL_SSL_CTX_set_cipher_list := LoadSSLFunction('SSL_CTX_set_cipher_list');
     @OpenSSL_SSL_CTX_set_default_passwd_cb := LoadSSLFunction('SSL_CTX_set_default_passwd_cb');
     @OpenSSL_SSL_CTX_set_default_passwd_cb_userdata := LoadSSLFunction('SSL_CTX_set_default_passwd_cb_userdata');
     @OpenSSL_SSL_CTX_load_verify_locations := LoadSSLFunction('SSL_CTX_load_verify_locations');
     @OpenSSL_SSL_CTX_use_PrivateKey_file := LoadSSLFunction('SSL_CTX_use_PrivateKey_file');
     @OpenSSL_SSL_CTX_use_certificate_file := LoadSSLFunction('SSL_CTX_use_certificate_file');
     @OpenSSL_SSL_CTX_check_private_key := LoadSSLFunction('SSL_CTX_check_private_key');

     @OpenSSL_SSL_new := LoadSSLFunction('SSL_new');
     @OpenSSL_SSL_free := LoadSSLFunction('SSL_free');
     @OpenSSL_SSL_accept := LoadSSLFunction('SSL_accept');
     @OpenSSL_SSL_connect := LoadSSLFunction('SSL_connect');
     @OpenSSL_SSL_pending := LoadSSLFunction('SSL_pending');
     @OpenSSL_SSL_read := LoadSSLFunction('SSL_read');
     @OpenSSL_SSL_write := LoadSSLFunction('SSL_write');
     @OpenSSL_SSL_shutdown := LoadSSLFunction('SSL_shutdown');
     @OpenSSL_SSL_set_fd := LoadSSLFunction('SSL_set_fd');
     @OpenSSL_SSL_set_shutdown := LoadSSLFunction('SSL_set_shutdown');
     @OpenSSL_SSL_get_version := LoadSSLFunction('SSL_get_version');
     @OpenSSL_SSL_get_current_cipher := LoadSSLFunction('SSL_get_current_cipher');
     @OpenSSL_SSL_get_error := LoadSSLFunction('SSL_get_error');

     @OpenSSL_SSL_CIPHER_get_bits := LoadSSLFunction('SSL_CIPHER_get_bits');
     @OpenSSL_SSL_CIPHER_get_name := LoadSSLFunction('SSL_CIPHER_get_name');
     @OpenSSL_SSL_CIPHER_description := LoadSSLFunction('SSL_CIPHER_description');

     @OpenSSL_X509_get_subject_name := LoadFunction('X509_get_subject_name');
     @OpenSSL_X509_NAME_oneline := LoadFunction('X509_NAME_oneline');
     @OpenSSL_X509_verify_cert_error_string := LoadFunction('X509_verify_cert_error_string');
     @OpenSSL_X509_STORE_CTX_get_current_cert := LoadFunction('X509_STORE_CTX_get_current_cert');
     @OpenSSL_X509_STORE_CTX_get_error := LoadFunction('X509_STORE_CTX_get_error');
     @OpenSSL_X509_STORE_CTX_get_error_depth := LoadFunction('X509_STORE_CTX_get_error_depth');
     @OpenSSL_X509_STORE_get_by_subject := LoadFunction('X509_STORE_get_by_subject');

     @OpenSSL_ERR_get_error := Loadfunction('ERR_get_error');
     @OpenSSL_ERR_error_string := Loadfunction('ERR_error_string');
     @OpenSSL_ERR_remove_state := Loadfunction('ERR_remove_state');


     if ( @OpenSSL_SSL_load_error_strings = nil )
        or ( @OpenSSL_SSL_library_init = nil )
        or ( @OpenSSL_SSLeay = nil )
        or ( @OpenSSL_SSLeay_version = nil )

        or ( @OpenSSL_RAND_load_file = nil )
        or ( @OpenSSL_RAND_write_file = nil )
        or ( @OpenSSL_RAND_screen = nil )
        or ( @OpenSSL_RAND_status = nil )
        or ( @OpenSSL_RAND_bytes = nil )  {PRNG}
        
        or ( @OpenSSL_SSLv23_client_method = nil )
        or ( @OpenSSL_SSLv23_server_method = nil )

        or ( @OpenSSL_SSL_CTX_new = nil )
        or ( @OpenSSL_SSL_CTX_free = nil )
        or ( @OpenSSL_SSL_CTX_ctrl = nil )
        or ( @OpenSSL_SSL_CTX_set_verify = nil )
        or ( @OpenSSL_SSL_CTX_set_cipher_list = nil )
        or ( @OpenSSL_SSL_CTX_load_verify_locations  = nil )
        or ( @OpenSSL_SSL_CTX_use_PrivateKey_file = nil )
        or ( @OpenSSL_SSL_CTX_check_private_key = nil )
        or ( @OpenSSL_SSL_CTX_use_certificate_file = nil )
        or ( @OpenSSL_SSL_CTX_set_default_passwd_cb = nil )
        or ( @OpenSSL_SSL_CTX_set_default_passwd_cb_userdata = nil )

        or ( @OpenSSL_SSL_new = nil )
        or ( @OpenSSL_SSL_free = nil )
        or ( @OpenSSL_SSL_accept = nil )
        or ( @OpenSSL_SSL_connect = nil )
        or ( @OpenSSL_SSL_pending = nil )
        or ( @OpenSSL_SSL_read = nil )
        or ( @OpenSSL_SSL_write = nil )
        or ( @OpenSSL_SSL_shutdown = nil )
        or ( @OpenSSL_SSL_set_fd = nil )
        or ( @OpenSSL_SSL_set_shutdown = nil )
        or ( @OpenSSL_SSL_get_version = nil )
        or ( @OpenSSL_SSL_get_current_cipher = nil )
        or ( @OpenSSL_SSL_get_error = nil )

        or ( @OpenSSL_SSL_CIPHER_get_bits = nil )
        or ( @OpenSSL_SSL_CIPHER_get_name = nil )
        or ( @OpenSSL_SSL_CIPHER_description = nil )

        or ( @OpenSSL_X509_get_subject_name = nil )
        or ( @OpenSSL_X509_NAME_oneline = nil )
        or ( @OpenSSL_X509_verify_cert_error_string = nil )
        or ( @OpenSSL_X509_STORE_CTX_get_current_cert = nil )
        or ( @OpenSSL_X509_STORE_CTX_get_error = nil )
        or ( @OpenSSL_X509_STORE_CTX_get_error_depth = nil )
        or ( @OpenSSL_X509_STORE_get_by_subject = nil )

        or ( @OpenSSL_ERR_get_error = nil )
        or ( @OpenSSL_ERR_error_string = nil )
        or ( @OpenSSL_ERR_remove_state = nil )

     then begin
        FreeLibrary( LibeayHandle );
        FreeLibrary( LibsslHandle );
        exit;
     end;

     if OpenSSL_SSLeay < REQUIRED_OPENSSL_VERSION then begin
        Log( LOGID_ERROR, Format(TrGl(kLog, 'SSL.OldVersion',
             'OpenSSL DLLs are too old (%s)'),
             [OpenSSL_SSLeay_version(SSLEAY_VERSION)]) );
        FreeLibrary( LibeayHandle );
        FreeLibrary( LibsslHandle );
        exit;
     end else begin
        Log( LOGID_DETAIL, OpenSSL_SSLeay_version( SSLEAY_VERSION ) )
     end;

     OpenSSL_SSL_load_error_strings;
     OpenSSL_SSL_library_init; // register the available ciphers and digests

     // Now we have to seed OpenSSL's pseudo random number generator
     Randfile := CfgIni.ReadString( 'SSL', 'RandFile', '' );
     if Randfile='' then Randfile := PATH_BASE + CFGFILE_RANDSEED;

     Bytes := OpenSSL_RAND_load_file( PChar(Randfile), MAX_RANDOM_BYTES );
     if Bytes = 0 then
        Log( LOGID_WARN, Format(TrGl(kLog, 'SSL.RandLoadError',
             'Failed to read random data from %s'), [Randfile]) )
     else
        Log( LOGID_DETAIL, Format(TrGl(kLog, 'SSL.RandLoad',
             'Read %d random bytes from %s'), [Bytes, Randfile]) );

     Bytes := OpenSSL_RAND_write_file( PChar(PATH_BASE + CFGFILE_RANDSEED) );
     if Bytes = -1 then
        Log( LOGID_WARN, Format(TrGl(kLog, 'SSL.RandWriteError',
             'Failed to write random data to %s'),
             [PATH_BASE + CFGFILE_RANDSEED]) )
     else
        Log( LOGID_DEBUG, Format(TrGl(kLog, 'SSL.RandWrite',
             'Wrote %d new random bytes to %s'),
             [Bytes, PATH_BASE + CFGFILE_RANDSEED]) );

     if( OpenSSL_RAND_status = 1 ) then
        Log( LOGID_DETAIL, TrGl(kLog, 'SSL.RandSuccess',
             'Pseudo random number generator seeded sucessfully') )
     else
        Log( LOGID_WARN, TrGl(kLog, 'SSL.RandFail',
             'Pseudo random number generator has not been seeded with enough data') );

     Result := True;
end;

procedure FreeSSL;
begin
     FreeLibrary( LibeayHandle );
     FreeLibrary( LibsslHandle );
     SSLReady := False;
end;

function VerifyCallback( preverify: Integer; x509_ctx: Pointer): Integer; cdecl;
var  err, depth: Integer;
     buf: Array[0..256] of Char;
     name: Pointer;
begin
     Result := 0;
     err   := OpenSSL_X509_STORE_CTX_get_error( x509_ctx );
     depth := OpenSSL_X509_STORE_CTX_get_error_depth( x509_ctx );
     name  := OpenSSL_X509_get_subject_name(
                 OpenSSL_X509_STORE_CTX_get_current_cert( x509_ctx ) );
     OpenSSL_X509_NAME_oneline( name, buf, 256);
     if preverify = 0 then
        Log( LOGID_WARN, Format(TrGl(kLog, 'SSL.VerifyError',
             'Verify error: %s depth=%d %s'),
             [OpenSSL_X509_verify_cert_error_string(err), depth, buf]) )
     else begin
        Log( LOGID_INFO, Format(TrGl(kLog, 'SSL.VerifyOk',
             'Verify ok: depth=%d %s'), [depth, buf]) );
        Result := 1;
     end;
end;

function VerifyCallbackLevel3( preverify: Integer; x509_ctx: Pointer): Integer; cdecl;
var  err, depth: Integer;
     buf: Array[0..256] of Char;
     name: Pointer;
begin
     Result := 0;
     err   := OpenSSL_X509_STORE_CTX_get_error( x509_ctx );
     depth := OpenSSL_X509_STORE_CTX_get_error_depth( x509_ctx );
     name  := OpenSSL_X509_get_subject_name(
                 OpenSSL_X509_STORE_CTX_get_current_cert( x509_ctx ) );
     OpenSSL_X509_NAME_oneline( name, buf, 256);
     if preverify = 0 then
        Log( LOGID_WARN, Format(TrGl(kLog, 'SSL.VerifyError',
             'Verify error: %s depth=%d %s'),
             [OpenSSL_X509_verify_cert_error_string(err), depth, buf]) )
     else begin
        if depth = 0 then begin
           if OpenSSL_X509_STORE_get_by_subject( x509_ctx, 1, name, x509_ctx ) <> 1 then begin
              Log( LOGID_WARN, Format(TrGl(kLog, 'SSL.VerifyLevel3Error',
                   'Verify error: peer certificate not installed locally: depth=%d %s'),
                   [depth, buf]) )
           end else begin
              Log( LOGID_INFO, Format(TrGl(kLog, 'SSL.VerifyLevel3Ok',
                   'Verify ok: peer certificate installed locally: depth=%d %s'),
                   [depth, buf]) );
              Result := 1;
           end
        end else begin
           Log( LOGID_INFO, Format(TrGl(kLog, 'SSL.VerifyOk',
                'Verify ok: depth=%d %s'), [depth, buf]) );
           Result := 1;
        end
     end;
end;

// PasswordCallback hands back the password to be used
// when loading/storing a PEM certificate with encryption.
function PasswordCallback( buf: PChar; size, rwflag: Integer;
                           userdata: Pointer): Integer; cdecl;
var  User, Pass: String;
     len, i: Integer;
begin
     PasswordFile.UsePassword( KEYID, User, Pass );

     len := Length( Pass );
     if len >= size then begin
        Log(LOGID_WARN, Format(TrGl(kLog, 'SSL.PasswordTooLong',
            'Certificate password length (%d) exceeds maximum (%d) - ' +
            'password will be truncated!'), [len, size-1]) );
        len := size;
     end;

     buf[len] := #0;
     for i:=len downto 1 do buf[i-1] := Pass[i];
     Result := len;
end;

// ---------------------------------------------------------- TSSLContext -----

procedure TSSLContext.SetOptions( CipherList: String ); {MG}{CipherList}
begin
     Log( LOGID_DEBUG, Self.Classname + '.SetOptions mode='
          + IntToStr( FVerify ) + ' options=' + IntToHex( FOptions, 8 )
          + ' ciphers=' + CipherList + ' verify locations='  {MG}{CipherList}
          + string(FCaFile) + ' ' + string(FCaPath) );

     OpenSSL_SSL_CTX_ctrl(Context, SSL_CTRL_OPTIONS, FOptions, nil );

     OpenSSL_SSL_CTX_set_cipher_list( Context, PChar( CipherList ) );  {MG}{CipherList}

     OpenSSL_SSL_CTX_set_default_passwd_cb( Context, @PasswordCallback );

     if FVerify > SSL_VERIFY_NONE then begin
        if FVerify >= SSL_VERIFY_ONLY_LOCAL_CERTS then
           OpenSSL_SSL_CTX_set_verify( Context, FVerify - SSL_VERIFY_ONLY_LOCAL_CERTS,
                                       @VerifyCallbackLevel3 )
        else
           OpenSSL_SSL_CTX_set_verify( Context, FVerify, @VerifyCallback );

        if ( FCaFile <> nil ) or ( FCaPath <> nil ) then begin
           if OpenSSL_SSL_CTX_load_verify_locations( Context, FCaFile, FCaPath ) <= 0
           then Log( LOGID_WARN, TrGl(kLog, 'SSL.LoadVerifyError',
                     'Loading SSL verify locations failed') );
        end;
     end;

     // Allow SSL_write(..., n) to return r with 0 < r < n (i.e. report success
     // when just a single record has been written). When not set (the default),
     // SSL_write() will only report success once the complete chunk was written.
     OpenSSL_SSL_CTX_ctrl( Context, SSL_CTRL_MODE, SSL_MODE_ENABLE_PARTIAL_WRITE, nil );

     // Never bother the application with retries if the transport is blocking.
     // [...] The flag SSL_MODE_AUTO_RETRY will cause read/write operations to
     // only return after the handshake and successful completion.
     OpenSSL_SSL_CTX_ctrl( Context, SSL_CTRL_MODE, SSL_MODE_AUTO_RETRY, nil );
end;

function TSSLContext.LoadPrivateKeyPair( AKey: PChar ): Boolean;
begin
     Log( LOGID_DEBUG, Self.Classname + '.LoadPrivateKeyPair ' + String(AKey) );
     Result := False;
     if OpenSSL_SSL_CTX_use_certificate_file( Context, AKey, 1 ) <> 1 then begin
        GetError( 'SSL_CTX_use_certificate_file' );
        exit;
     end;
     if OpenSSL_SSL_CTX_use_PrivateKey_file( Context, AKey, 1 ) <> 1 then begin
        GetError( 'SSL_CTX_use_PrivateKey_file' );
        exit;
     end;
     if OpenSSL_SSL_CTX_check_private_key( Context ) <> 1 then
        GetError( 'SSL_CTX_check_private_key' )
     else Result := True
end;

constructor TSSLContext.Create( AServer: Boolean; AVerify: Integer;
                                ACaFile, ACaPath: String );
begin
     inherited Create;
     Log( LOGID_DEBUG, Self.Classname + '.Create' );
     FVerify := AVerify;
     if ACaFile <> '' then FCaFile := PChar( ACaFile ) else FCaFile := nil;
     if ACaPath <> '' then FCaPath := PChar( ACaPath ) else FCaPath := nil;

     if Aserver and CfgIni.ReadBool( 'SSL', 'UseServerModeV2', False ) then
        FOptions := SSL_OP_ALL
     else
        FOptions := SSL_OP_NO_SSLv2 or SSL_OP_ALL;
     if not Def_UseSSLv3 then FOptions := FOptions or SSL_OP_NO_SSLv3;
     if not Def_UseTLSv1 then FOptions := FOptions or SSL_OP_NO_TLSv1;

     if AServer then
        FContext := OpenSSL_SSL_CTX_new( OpenSSL_SSLv23_server_method )
     else
        FContext := OpenSSL_SSL_CTX_new( OpenSSL_SSLv23_client_method );
     if FContext = nil then GetError( 'SSL_CTX_new' );
end;

destructor TSSLContext.Destroy;
begin
     Log( LOGID_DEBUG, Self.Classname + '.Destroy' );
     if Context <> nil then OpenSSL_SSL_CTX_free( Context );
     inherited Destroy;
end;

// ------------------------------------------------------- TSSLConnection -----

function TSSLConnection.Connect( const ASocketHandle: Integer ): Boolean;
begin
     Log( LOGID_DEBUG, Self.Classname + '.Connect' );
     Result := False;
     if Ssl = nil then exit;
     OpenSSL_SSL_set_fd( Ssl, ASocketHandle );
     if OpenSSL_SSL_connect( Ssl ) <= 0
        then GetError( 'SSL_connect' )
        else Result := True;
end;

function TSSLConnection.Accept( const ASocketHandle: Integer ): Boolean;
begin
     Log( LOGID_DEBUG, Self.Classname + '.Accept' );
     Result := False;
     if Ssl = nil then exit;
     OpenSSL_SSL_set_fd( Ssl, ASocketHandle );
     if OpenSSL_SSL_accept( Ssl ) <= 0 then GetError( 'SSL_accept' )
     else Result := True;
end;

procedure TSSLConnection.Info;
var  Cipher: Pointer;
     TotalBits, SecretBits: Integer;
begin
     Log( LOGID_DEBUG, Self.Classname + '.Info' );
     Cipher := OpenSSL_SSL_get_current_cipher( Ssl );
     SecretBits := OpenSSL_SSL_CIPHER_get_bits( Cipher, @TotalBits );
     Log( LOGID_INFO, Format(TrGl(kLog, 'SSL.ConnectionInfo',
          'Secure connection with %s, cipher %s, %d secret bits (%d total)'),
          [OpenSSL_SSL_get_version( Ssl ), OpenSSL_SSL_CIPHER_get_name( Cipher ),
          SecretBits, TotalBits]) );
end;

function TSSLConnection.HasPendingData: Boolean;
begin
     if OpenSSL_SSL_pending( Ssl ) > 0 then Result := True
     else Result := False;
end;

function TSSLConnection.Read( Buf: PChar; Num: Integer): Integer;
var  ret: Integer;
begin
     Result := 0;
     ret := OpenSSL_SSL_read( Ssl, Buf, Num );
     case OpenSSL_SSL_get_error( Ssl, ret ) of
        SSL_ERROR_NONE:
           Result := ret;
        SSL_ERROR_ZERO_RETURN:
           Log( LOGID_WARN, Self.Classname + '.Read ' +
                TrGl(kLog, 'SSL.ConnectionClosed', 'SSL connection closed' ) );
        SSL_ERROR_SYSCALL:
           if ret < 0 then GetError( 'SSL_read (SSL_ERROR_SYSCALL)' )
           else Log( LOGID_ERROR, Self.Classname + '.Read ' +
                     TrGl(kLog, 'SSL.EOF', 'unexpected end of file' ) );
        SSL_ERROR_SSL:
           GetError( 'SSL_read' );
     end;
end;

function TSSLConnection.Write( Buf: PChar; Num: Integer): Integer;
var  ret: Integer;
begin
     Result := -1;
     ret := OpenSSL_SSL_write( Ssl, Buf, Num );
     case OpenSSL_SSL_get_error( Ssl, Ret ) of
        SSL_ERROR_NONE:
           Result := ret;
        SSL_ERROR_ZERO_RETURN:
           Log( LOGID_WARN, Self.Classname + '.Write ' +
                TrGl(kLog, 'SSL.ConnectionClosed', 'SSL connection closed' ) );
        SSL_ERROR_SYSCALL:
           if ret < 0 then GetError( 'SSL_write (SSL_ERROR_SYSCALL)' )
           else Log( LOGID_ERROR, Self.Classname + '.Write ' +
                     TrGl(kLog, 'SSL.EOF', 'unexpected end of file' ) );
        SSL_ERROR_SSL:
           GetError( 'SSL_write' );
     end;
end;

procedure TSSLConnection.Shutdown;
begin
     Log( LOGID_DEBUG, Self.Classname + '.Shutdown' );
     OpenSSL_SSL_shutdown( Ssl );
end;

constructor TSSLConnection.Create( ASSLContext: Pointer );
begin
   inherited Create;
   Log( LOGID_DEBUG, Self.Classname + '.Create' );
   FSsl := OpenSSL_SSL_new( ASSLContext );
   if FSsl = nil then GetError( 'SSL_new' );
end;

destructor TSSLConnection.Destroy;
begin
     Log( LOGID_DEBUG, Self.Classname + '.Destroy' );
     try
        OpenSSL_SSL_set_shutdown( Ssl, SSL_SENT_SHUTDOWN or SSL_RECEIVED_SHUTDOWN );
        OpenSSL_SSL_free( Ssl );
        OpenSSL_ERR_remove_state( 0 );
     except end;
     inherited Destroy;
end;

// ----------------------------------------------------------------------------

Procedure FillSSLModeList (List: TStrings);
begin
   With List do begin
      Clear;
      Add ( TrGl ( 'SSL', 'SSLMODE_DontUseSSL', 'Don''t use SSL' ) );
      Add ( TrGl ( 'SSL', 'SSLMODE_UseAlwaysOnSecurePort', 'Always use SSL on secure port' ) );
      Add ( TrGl ( 'SSL', 'SSLMODE_NegotiateIfPossible', 'Negotiate SSL, if possible' ) );
      Add ( TrGl ( 'SSL', 'SSLMODE_NegotiateAlways', 'Negotiate SSL always' ) )
   end
end;

Procedure FillSSLVerifyModeList (List: TStrings);
begin
   With List do begin
      Clear;
      Add ( TrGl ( 'SSL', 'SSLVerify_None', 'No verify' ) );
      Add ( TrGl ( 'SSL', 'SSLVerify_PeerCertificateIfPresent', 'Verify peer certificate if present' ) );
      Add ( TrGl ( 'SSL', 'SSLVerify_AlwaysVerify', 'Always verify peer certificate' ) );
      Add ( TrGl ( 'SSL', 'SSLVerify_VerifyWithLocalCertifs', 'Verify peer with locally installed certificates' ) )
   end
end;

function SSL_RandBytes( out Buf: Array of Char; Bytes: Integer ): Boolean;
begin
   Result := False;
   FillChar( Buf, SizeOf(Buf), 0 );
   if Bytes > SizeOf(Buf) then exit;
   if OpenSSL_RAND_bytes( Buf, Bytes ) = 1 then Result := True
   else GetError( 'RandBytes' )
end;

function OpenSSLVersion: Integer; {MG}{AES}
begin
     Result := OpenSSL_SSLeay;
end;

end.
