// ============================================================================
// WinSock-related tools.
// Copyright (c) 1999, Juergen Haible. All Rights Reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.
// ============================================================================

unit uWinSock; // WinSock-related tools

// ----------------------------------------------------------------------------
// Contains WinSock-related tools
// ----------------------------------------------------------------------------

interface

function nAddrToStr( nAddr: LongInt ): String;
function hAddrToStr( hAddr: LongInt ): String;
function LookupLocalHostName: String;
function LookupLocalHostAddr: LongInt;
function LookupHostName( HostAddr: LongInt ): String;
function LookupHostAddr( Hostname: String ): LongInt;
function IsLocalHost( nAddr: LongInt ): Boolean;
function WinsockErrTxt( ErrNo: LongInt ): String;
{JW} {HOSTS}
Function GetHostsDirectory(Var Path: String): Boolean;
function SetHostsEntry_ByName(Hostname,HostAdresse:String):Integer;
function SetHostsEntry_ByAddress(Hostname,HostAdresse:String):Integer;
{JW} {delete hostsentry}
function DeleteHostsEntry(Hostname,HostAdresse:String):Integer;
function StrToAddr(AddressString:String): Longint;
{JW}
// ----------------------------------------------------------------------------

implementation

uses SysUtils, WinSock, uTools, classes;

{JW} {HOSTS}

Function GetHostsDirectory(Var Path: String): Boolean;
begin
   Result := true;
   If FileExists2(GetWindowsPath+'hosts.') then Path:=GetWindowsPath
   else if FileExists2(GetSystemPath+'hosts.') then Path:=GetSystemPath
   else if FileExists2(GetSystemPath+'drivers\etc\hosts.') then Path:=GetSystemPath+'drivers\etc\'
   else begin
      Result:=False;  // hosts notfound
      If IsWindowsNT
         then Path := GetSystemPath+'drivers\etc\hosts.'
         else Path := GetWindowsPath
   end
end;

Procedure SplitHostEntry (Const Entry: String; Var Adr, Nam: String);
Var s: String; c: Char; M, i: Integer;
begin
   Adr := '';
   Nam := '';
   s := Trim(Entry);
   If s > '' then begin
      If s[1]=';' then Exit;
      M := 0;
      For i := 1 to Length(s) do begin
         c := s[i];
         Case M of
            0: If c IN['0'..'9', '.'] then Adr := Adr + c
               else If c IN[#9, ' '] then Inc(M)
               else Exit;
            1: If Not (c IN[#9, ' ']) then begin Nam := c; Inc(M) end;
            2: If Not (c IN[#9, ' ']) then Nam := Nam + c
               else Exit;
         end
      end
   end
end;

function SetHostsEntry_byName(HostName,HostAdresse:String):Integer;
var
   Hostspath, Adr, Nam: String;
   HostsFound    : Boolean;
   i, p          : Integer;
begin
   result:=0;
   if Hostname='' then begin
      result:=-2;
      exit;
   end;
   HostsFound := GetHostsDirectory (Hostspath);
   With TStringList.Create do try
      if HostsFound then LoadFromFile(HostsPath+'hosts');
      p := -1;
      for i:=0 to Count-1 do begin
         SplitHostEntry (Strings[i], Adr, Nam);
         If Nam = Trim(HostName) then begin
            p := i; break
         end
      end;
      If p >= 0 then begin
         if HostAdresse=''
            then Delete(p)
            else Strings[p]:=HostAdresse+' '+HostName+' #Modified by Hamster';
      end else begin
         if HostAdresse <> ''
            then Add(HostAdresse+' '+HostName+' #Created by Hamster')
            else Result:=-1
      end;
      SaveToFile(HostsPath+'hosts')
   finally
      Free
   end
end;

function SetHostsEntry_byAddress(HostName,HostAdresse:String):Integer;
var
   Hostspath, Adr, Nam: String;
   HostsFound    : Boolean;
   i, p          : Integer;
begin
   result:=0;
   if HostAdresse = '' then begin
      result:=-2;
      exit;
   end;
   HostsFound := GetHostsDirectory (Hostspath);
   With TStringList.Create do try
      if HostsFound then LoadFromFile(HostsPath+'hosts');
      p := -1;
      for i:=0 to Count-1 do begin
         SplitHostEntry (Strings[i], Adr, Nam);
         If Adr = Trim(HostAdresse) then begin
            p := i; break
         end
      end;
      If p >=0 then begin
         if HostName ='' then Delete(p)
                        else Strings[p]:=HostAdresse+' '+HostName+' #Modified by Hamster';
      end else begin
         if HostAdresse<>'' then Add(HostAdresse+' '+HostName+' #Created by Hamster')
                            else Result:=-1
      end;
      SaveToFile(HostsPath+'hosts')
   finally
      Free;
   end;
end;

{JW} {delete hostsentry}
function DeleteHostsEntry(Hostname,HostAdresse:String):Integer;
var
    Hostspath, Adr, Nam: String;
    i, p          : Integer;
begin
   result:=0;
   if (Hostname='') or (HostAdresse='') then begin
      result:=-2;
      exit;
   end;
   If GetHostsDirectory (Hostspath) then begin
      With TStringList.Create do try
         LoadFromFile(HostsPath+'hosts');
         p := -1;
         for i:=0 to Count-1 do begin
            SplitHostEntry (Strings[i], Adr, Nam);
            If (Nam = Trim(HostName)) and (Adr = Trim(HostAdresse)) then begin
               p := i; break
            end
         end;
         If p >= 0 then begin
            Delete(p);
            SaveToFile(HostsPath+'hosts')
         end else begin
            Result := -1
         end
      finally
         Free
      end
   end
end;
{JW}

function StrToAddr(AddressString:String): Longint;
Var x, l, Error, p, i: Integer;
begin
  result:=-1;
  x := 0;
  For i := 1 To 4 do begin
     p := pos('.',AddressString);
     If (i < 4) and (p=0) then Exit;
     If (i = 4) and (p>0) then Exit;
     If p > 0 then begin
        Val (copy(AddressString, 1, p-1), l, Error);
        Delete (AddressString, 1, p)
     end else begin
        Val (AddressString, l, Error);
     end;
     if Error <> 0 then exit;
     If (l < 0) or (l > 255) then Exit;
     Case i of
       1: x := l;
       2: x := x + (l shl 8);
       3: x := x + (l shl 16);
       4: x := x + (l shl 24)
     end
  end;
  Result := x
end;

function nAddrToStr( nAddr: LongInt ): String;
begin
     Result := hAddrToStr( ntohl( nAddr ) );
end;

function hAddrToStr( hAddr: LongInt ): String;
begin
     Result := inttostr( (hAddr shr 24)         ) + '.'
             + inttostr( (hAddr shr 16) and $ff ) + '.'
             + inttostr( (hAddr shr 8 ) and $ff ) + '.'
             + inttostr( (hAddr       ) and $ff );
end;

function LookupLocalHostName: String;
const SIZE=255;
var  HostNam: Pointer;
     HostEnt: PHostEnt;
begin
     Result:='localhost';
     GetMem( HostNam, SIZE );

     try
        if GetHostName( HostNam, SIZE ) = 0 then begin
           if PChar(HostNam)^<>#0 then begin
              HostEnt := GetHostByName( HostNam );
              if HostEnt<>nil then Result := String( HostEnt^.h_name );
           end;
        end;
     finally
       FreeMem( HostNam, SIZE );
     end;
end;

function LookupLocalHostAddr: LongInt;
const SIZE=255;
var  HostNam: Pointer;
     HostEnt: PHostEnt;
     Name   : Pointer; //JW //"HostAddrNew1"
     i:Integer;
begin
     //  Result := $7F000001; // 127.0.0.1
     Result:=htonl($7F000001); //HRR


     GetMem( HostNam, SIZE );

     try
        i:=GetHostName( HostNam, SIZE );
        if i = 0 then begin
           if PChar(HostNam)^<>#0 then begin
              HostEnt := GetHostByName( HostNam );
              if HostEnt<>nil then begin
//                 Result := LongInt( Pointer( HostEnt^.h_addr_list^ )^ );
{JW} {"HostAddrNew1"}
                  Name:=HostEnt^.h_addr^;
                  Result := LongInt( name^);
{/JW}
              end;
           end;
        end;
     finally
        FreeMem( HostNam, SIZE );
     end;
end;

function IsLocalHost( nAddr: LongInt ): Boolean;
const SIZE=255;
var  HostNam: Pointer;
     HostEnt: PHostEnt;
     AddrList: Pointer;
begin
     Result := False;
     if nAddr=$0100007f then begin Result:=True; exit; end;

     GetMem( HostNam, SIZE );
     try
        if GetHostName( HostNam, SIZE ) = 0 then begin
           if PChar(HostNam)^<>#0 then begin
              HostEnt := GetHostByName( HostNam );
              if HostEnt<>nil then begin
                 try
                    AddrList := Pointer( HostEnt^.h_addr_list^ );
                    while (LongInt(AddrList)<>0) and (LongInt(AddrList^)<>0) do begin
                       if LongInt( AddrList^ )=nAddr then begin Result:=True; break; end;
                       AddrList := Pointer( LongInt(AddrList) + sizeof(LongInt) );
                    end;
                 except
                 end;
              end;
           end;
        end;
     finally
        FreeMem( HostNam, SIZE );
     end;
end;

function LookupHostAddr( Hostname: String ): LongInt;
var
   HostEnt: PHostEnt;
   Name   : Pointer; //JW //HostAddrNew
begin
     Result := 0;
     if HostName='' then exit;

     try
        Result := inet_addr( PChar(HostName) );
        if Result=LongInt(INADDR_NONE) then begin
           HostEnt := GetHostByName( PChar(HostName) );
           if HostEnt<>nil then begin
{JW} {AddrKorr}
               Name:=HostEnt^.h_addr^;
               Result := LongInt( name^);
{/JW}
           end else begin
              Result := 0; 
           end;
        end;
     except
        Result := 0;
     end;
end;

function LookupHostName( HostAddr: LongInt ): String;
var  HostEnt: PHostEnt;
begin
     Result := '[' + nAddrToStr( HostAddr ) + ']';

     try
        HostEnt := GetHostByAddr( @HostAddr, 4, PF_INET );
        if HostEnt<>nil then Result := String( HostEnt^.h_name );
     except
     end;
end;

function WinsockErrTxt( ErrNo: LongInt ): String;
begin
     case ErrNo of
        10004: Result := 'WSAEINTR';
        10009: Result := 'WSAEBADF';
        10013: Result := 'WSEACCES';
        10014: Result := 'WSAEFAULT';
        10022: Result := 'WSAEINVAL';
        10024: Result := 'WSAEMFILE';
        10035: Result := 'WSAEWOULDBLOCK';
        10036: Result := 'WSAEINPROGRESS';
        10037: Result := 'WSAEALREADY';
        10038: Result := 'WSAENOTSOCK';
        10039: Result := 'WSAEDESTADDRREQ';
        10040: Result := 'WSAEMSGSIZE';
        10041: Result := 'WSAEPROTOTYPE';
        10042: Result := 'WSAENOPROTOOPT';
        10043: Result := 'WSAEPROTONOSUPPORT';
        10044: Result := 'WSAESOCKTNOSUPPORT';
        10045: Result := 'WSAEOPNOTSUPP';
        10046: Result := 'WSAEPFNOSUPPORT';
        10047: Result := 'WSAEAFNOSUPPORT';
        10048: Result := 'WSAEADDRINUSE';
        10049: Result := 'WSAEADDRNOTAVAIL';
        10050: Result := 'WSAENETDOWN';
        10051: Result := 'WSAENETUNREACH';
        10052: Result := 'WSAENETRESET';
        10053: Result := 'WSAECONNABORTED';
        10054: Result := 'WSAECONNRESET';
        10055: Result := 'WSAENOBUFS';
        10056: Result := 'WSAEISCONN';
        10057: Result := 'WSAENOTCONN';
        10058: Result := 'WSAESHUTDOWN';
        10059: Result := 'WSAETOOMANYREFS';
        10060: Result := 'WSAETIMEDOUT';
        10061: Result := 'WSAECONNREFUSED';
        10062: Result := 'WSAELOOP';
        10063: Result := 'WSAENAMETOOLONG';
        10064: Result := 'WSAEHOSTDOWN';
        10065: Result := 'WSAEHOSTUNREACH';
        10091: Result := 'WSASYSNOTREADY';
        10092: Result := 'WSAVERNOTSUPPORTED';
        10093: Result := 'WSANOTINITIALISED';
        10101: Result := 'WSAEDISCON';
        11001: Result := 'WSAHOST_NOT_FOUND';
        11002: Result := 'WSATRY_AGAIN';
        11003: Result := 'WSANO_RECOVERY';
        11004: Result := 'WSANO_DATA';
        else Result:='';
     end;
end;

// ----------------------------------------------------------------------------

end.
