unit uHeader;

interface

Uses classes, uCSMap;

Type
  THeaderinhalt = (hiDecoded, hiRaw);

  TChangedStringlist = Class(TStrings)
  private
     FL: TStringlist;
  protected
     Procedure Put (i: Integer; Const S: String); override;
     Function  Get (i: Integer): String; override;
     Procedure SetVal (v: String; S: String); virtual;
     Function  GetVal (v: String): String; virtual;
     Procedure SetAll (Const S: String); virtual;
     Function  GetAll: String; virtual;
     Procedure PutObject (i: Integer; O: TObject); override;
     Function  GetObject (i: Integer): TObject; override;
     Function  GetCount: Integer; override;
  public
     Changed: boolean;
     Constructor Create;
     Destructor Destroy; override;
     Property Count: Integer Read GetCount;
     Procedure Clear; override;
     Property Strings[i: Integer]: String Read Get Write Put; default;
     Property Values[v: String]: String Read GetVal Write SetVal;
     Property Objects[i: Integer]: TObject Read GetObject Write PutObject;
     Function Add (Const s: String): Integer; override;
     Function AddObjects (Const s: String; O: TObject): Integer; virtual;
     Procedure AddStrings (sl: TStrings); override;
     Procedure Assign (Source: TPersistent); override;
     Procedure Exchange (i, j: Integer); override;
     Property Text: String Read GetAll Write SetAll;
     Procedure Delete (i: Integer); override;
     Procedure Insert (i: Integer; Const s: String); override;
     function  IndexOf(const S: string): Integer; override;
  end;

  THeader = Class (TChangedStringList)
    private
      fCharsets: TCharsets;
      Function GetName (i: Integer): String;
    public
      Constructor Create (Charsets: TCharsets);
      Property Charsets: TCharsets Read fCharsets;
      Property Name [i: Integer]: String Read GetName;
      Function Change (Const H, I: String): boolean;
      Function Position (Const H: String): Integer;
      Function Add (Const s: String): Integer; override;
      Function AddRaw (Const s: String): Integer;
      Function Inhalt (Const H: String; Const hi: THeaderinhalt): String;
      Function InhaltPerPos (Const i: Integer; Const hi: THeaderinhalt): String;
  end;

Function ToHeader (Const s: String): String;
Function Split_Headerzeile (Const Ges: String; Var Header, Inhalt: String): boolean;
Function TestHeadername (Const s: String; Var p: Integer): Boolean;
//Function HeaderDecode (Const p: Integer; Const InclHeader: boolean): String;

implementation

Uses SysUtils, uISO, Settings, uTransla;

{ ----------------------------------------------------------------------------- }

Const
  HeaderCont = [^I, ' '];
  Headerzeichen = [#33..#128];
Var
  slSonderheader: TStringlist;

Function ToHeader (Const s: String): String;
begin
   Result := Trim(s);
   If Result = '' then Exception.Create(TrL('FunctionToHeader-HeadernameEmpty',
      'bergebener Headername ist leer!'));
   If Result[Length(Result)] <> ':' then Result := Result + ':';
   Result := Result + ' '
end;

Function Split_Headerzeile (Const Ges: String; Var Header, Inhalt: String): boolean;
Var p, q: Integer; s: String;
begin
   Result := false; s := Ges;
   If (s='') or (s[1]=' ') then exit;
   p := Pos(':', s); q := Pos('=', s);
   If (p=0) and (q>0) then p:=q;
   While (p>2) and (s[p-1] IN HeaderCont) do begin
      s := Copy(s, 1, p-2) + Copy(s, p, Length(s)-p+1); Dec(p)
   end;
   If (p > 1) then begin
      Header := Copy (s, 1, p-1);
      If (Length(s)>p) and (s[p+1]<>' ')
         then Inhalt := Copy(s, p+1, Length(s)-p)
         else Inhalt := Copy(s, p+2, Length(s)-p-1);
      Result := true
   end
end;

Function TestHeadername (Const s: String; Var p: Integer): Boolean;
Var i: Integer;
begin
   Result := true;
   p := 0;
   If slSonderheader.IndexOf(s) >= 0 then exit;
   For i:=1 to Length(s) do begin
      If Not (s[i] IN Headerzeichen) then begin
         Result := false; p := i; exit
      end
   end
end;

{ ----------------------------------------------------------------------------- }

Constructor TChangedStringlist.Create;
begin
   inherited create;
   FL := TStringlist.Create
end;

Destructor TChangedStringlist.Destroy;
begin
   FL.free;
   inherited destroy;
end;

Function TChangedStringlist.GetCount: Integer;
begin
   Result := FL.Count
end;

Procedure TChangedStringlist.SetAll (Const S: String);
begin
   FL.text := S; changed := true
end;

Function TChangedStringlist.GetAll: String;
begin
   Result := FL.text
end;

Procedure TChangedStringlist.SetVal (v: String; S: String);
begin
   FL.Values[v] := s;
   changed := true
end;

Function  TChangedStringlist.GetVal (v: String): String;
begin
   Result := Fl.Values[v]
end;

Procedure TChangedStringlist.PutObject (i: Integer; O: TObject);
begin
   Fl.Objects[i] := O
end;

Function  TChangedStringlist.GetObject (i: Integer): TObject;
begin
   Result := Fl.Objects[i]
end;

Procedure TChangedStringlist.Clear;
begin
   If FL.Count > 0 then begin
      FL.Clear; changed := true
   end
end;

Function TChangedStringlist.Add (Const s: String): Integer;
begin
   Result := FL.Add(s);
   changed := true
end;

Function TChangedStringlist.AddObjects (Const s: String; O: TObject): Integer;
begin
   Result := FL.AddObject(s, O);
   changed := true
end;

Procedure TChangedStringlist.AddStrings (sl: TStrings);
begin
   If sl.Count > 0 then begin
      FL.AddStrings(sl);
      changed := true
   end
end;

Procedure TChangedStringlist.Assign (Source: TPersistent);
begin
   FL.Assign (Source);
   changed := true
end;

Procedure TChangedStringlist.Exchange (i, j: Integer);
begin
   FL.Exchange (i, j); changed := true
end;

Procedure TChangedStringlist.Delete (i: Integer);
begin
   Fl.Delete(i);
   changed := true
end;

Procedure TChangedStringlist.Put (i: Integer; Const S: String);
begin
   If s <> FL[i] then begin
      FL[i] := s; changed := true
   end
end;

Function TChangedStringlist.Get (i: Integer): String;
begin
   Result := FL[i]
end;

Procedure TChangedStringlist.Insert (i: Integer; Const s: String);
begin
   FL.Insert(i, s);
   changed := true
end;

function  TChangedStringlist.IndexOf(const S: string): Integer; 
begin
   Result := FL.IndexOf(s)
end;

{ ----------------------------------------------------------------------------- }

Constructor THeader.Create (Charsets: TCharsets);
begin
   inherited Create;
   fCharsets := Charsets
end;

Function THeader.GetName (i: Integer): String;
Var p: Integer;
begin
   p := Pos(': ', Strings[i]);
   If p>0 then Result := Copy(Strings[i],1,p-1)
          else Result := ''
end;

Function THeader.Change (Const H, I: String): boolean;
Var p: Integer; Inh: String;
begin
   Inh := EncodeISO (fCharsets, I);
   Result := true;
   p := Position(H);
   If Inh = '' then begin
      If P >= 0 then Delete(P)
                else Result := false
   end else begin
      If p >= 0 then begin
         If Strings[p] = Inh
            then Result := false
            else Strings[p] := ToHeader(H) + Inh
      end else begin
         Add (ToHeader(H) + Inh)
      end
   end
end;

Function THeader.Add (Const s: String): Integer;
begin
   Result := inherited Add(EncodeISO(fCharsets, s))
end;      

Function THeader.AddRaw (Const s: String): Integer;
begin
   Result := inherited Add(s)
end;      

Function THeader.Position (Const H: String): Integer;
Var i, L: Integer; s2: String;
begin
   Result := -1; s2 := LowerCase(ToHeader(H)); L := Length(s2);
   For i:=0 to Count-1 do
      If LowerCase(Copy(Strings[i], 1, L)) = s2
         then begin Result := i; break end
end;

Function THeader.Inhalt (Const H: String; Const hi: THeaderinhalt): String;
Var i: Integer;
begin
   i := Position(H);
   if i<0 then Result := '' else Result := InhaltPerPos(i, hi)
end;

Function THeader.InhaltPerPos (Const i: Integer; Const hi: THeaderinhalt): String;
Var p: Integer;
begin
   Result := Strings[i]; p := Pos(':', Result);
   Result := Copy(Result, p+2, Length(Result)-p-1);
   If hi = hiDecoded then Result := DecodeISO(fCharsets, Result)
end;

initialization
   slSonderheader := TStringlist.Create;
   LoadTextFile(ExtractFilePath(ParamStr(0))+'lHeader.txt', slSonderheader);
   With slSonderheader do If Count = 0 then begin
      Add ('!MAIL FROM');
      Add ('!RCPT TO');
      SaveToFile (ExtractFilePath(ParamStr(0))+'lHeader.txt')
   end;
finalization
   slSonderheader.free
end.
